package com.atlassian.marketplace.client.impl;

import java.io.InputStream;
import java.net.URI;

import com.atlassian.marketplace.client.MpacException;

import com.google.common.collect.Multimap;

/**
 * Abstraction of how to execute HTTP requests against the API.
 */
public interface HttpHelper
{
    public interface Response
    {
        /**
         * The status code of the response.
         */
        int getStatus();
        
        /**
         * An input stream for the response body.
         */
        InputStream getContentStream() throws MpacException;
        
        /**
         * True if the response has a zero-length body.
         */
        boolean isEmpty();
        
        /**
         * Ensures that the input stream is closed.
         */
        void close();
    }
    
    /**
     * Performs an HTTP GET.
     * @param uri  URI of resource to get
     * @return  a {@link Response} wrapping the HTTP response
     * @throws MpacException  if the server was unavailable or returned no response
     */
    Response get(URI uri) throws MpacException;
    
    /**
     * Performs an HTTP POST with form parameters.
     * @param uri  URI of resource to post to
     * @param params  the unencoded parameters
     * @return  a {@link Response} wrapping the HTTP response
     * @throws MpacException  if the server was unavailable or returned no response
     */
    Response postParams(URI uri, Multimap<String, String> params) throws MpacException;
    
    /**
     * Performs an HTTP PUT of an API entity.  The content type is assumed to be JSON.
     * @param uri  URI of resource to put
     * @param content  request body
     * @return  a {@link Response} wrapping the HTTP response
     * @throws MpacException  if the server was unavailable or returned no response
     */
    Response put(URI uri, byte[] content) throws MpacException;
    
    /**
     * Performs an HTTP DELETE.
     * @param uri  URI of resource to delete
     * @return  a {@link Response} wrapping the HTTP response
     * @throws MpacException  if the server was unavailable or returned no response
     */
    Response delete(URI uri) throws MpacException;
}
