package com.atlassian.marketplace.client.api;

import com.atlassian.upm.api.util.Option;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;

import static com.atlassian.upm.api.util.Option.none;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Encapsulates search parameters that can be passed to {@link Plugins#findBanners(BannerQuery)}.
 */
public final class BannerQuery
{
    private final Option<ApplicationKey> application;
    private final Option<Long> appBuildNumber;
    private final Option<String> label;
    private final int offset;
    private final Option<Integer> limit;
    
    /**
     * Returns a new {@link Builder} for constructing a BannerQuery.
     */
    public static Builder builder()
    {
        return new Builder();
    }

    public static Builder builder(BannerQuery query)
    {
        return builder()
            .application(query.getApplication())
            .appBuildNumber(query.getAppBuildNumber())
            .label(query.getLabel())
            .offset(query.getOffset())
            .limit(query.getLimit());
    }

    private BannerQuery(Builder builder)
    {
        application = builder.application;
        appBuildNumber = builder.appBuildNumber;
        label = builder.label;
        offset = builder.offset;
        limit = builder.limit;
    }
    
    public Option<ApplicationKey> getApplication()
    {
        return application;
    }

    public Option<Long> getAppBuildNumber()
    {
        return appBuildNumber;
    }

    public Option<String> getLabel()
    {
        return label;
    }
    
    public int getOffset()
    {
        return offset;
    }
    
    public Option<Integer> getLimit()
    {
        return limit;
    }

    @Override
    public String toString()
    {
        ImmutableList.Builder<String> params = ImmutableList.builder();
        for (ApplicationKey a: application)
        {
            params.add("application(" + a.getKey() + ")");
        }
        for (Long ab: appBuildNumber)
        {
            params.add("appBuildNumber(" + ab + ")");
        }
        for (String l: label)
        {
            params.add("label(" + l + ")");
        }
        if (offset > 0)
        {
            params.add("offset(" + offset + ")");
        }
        for (Integer l: limit)
        {
            params.add("limit(" + l + ")");
        }
        return "PluginQuery(" + Joiner.on(", ").join(params.build()) + ")";
    }
    
    @Override
    public boolean equals(Object other)
    {
        return (other instanceof BannerQuery) ? toString().equals(other.toString()) : false;
    }

    @Override
    public int hashCode()
    {
        return toString().hashCode();
    }
    
    public static class Builder
    {
        private Option<ApplicationKey> application = none();
        private Option<Long> appBuildNumber = none();
        private Option<String> label = none();
        private int offset = 0;
        private Option<Integer> limit = none();
        
        public BannerQuery build()
        {
            return new BannerQuery(this);
        }
        
        /**
         * Restricts the query to plugins that are compatible with the specified application.
         * @param application  an {@link ApplicationKey}, or {@link Option#none} for no application filter
         * @return  the same Builder
         */
        public Builder application(Option<ApplicationKey> application)
        {
            this.application = checkNotNull(application);
            return this;
        }
        
        /**
         * Restricts the query to plugins that are compatible with the specified application version.
         * This is ignored if you have not specified {@link #application}.
         * @param appBuildNumber  the application build number, or {@link Option#none} for no application
         *   version filter
         * @return  the same Builder
         */
        public Builder appBuildNumber(Option<Long> appBuildNumber)
        {
            this.appBuildNumber = checkNotNull(appBuildNumber);
            return this;
        }
        
        /**
         * Restricts the query to plugins that are marked with the specified label; this is a string
         * defined internally in the Atlassian Marketplace server, which is not visible in the UI.
         * @param label  the label to search for, or {@link Option#none} for no such filter
         * @return  the same Builder
         */
        public Builder label(Option<String> label)
        {
            this.label = label;
            return this;
        }

        /**
         * Specifies the number of items to skip ahead in the result set.
         * @param offset  the starting item index (zero to start at the beginning)
         * @return  the same Builder
         */
        public Builder offset(int offset)
        {
            if (offset < 0)
            {
                throw new IllegalArgumentException("offset may not be negative");
            }
            this.offset = offset;
            return this;
        }
        
        /**
         * Specifies the maximum number of items to return at a time.  You may receive fewer items
         * than this if you exceed the maximum number allowed by the server, or if there aren't that
         * many items. 
         * @param limit  the maximum number of items, or {@link Option#none} to use the server's default
         * @return  the same Builder
         */
        public Builder limit(Option<Integer> limit)
        {
            for (int l: limit)
            {
                if (l < 0)
                {
                    throw new IllegalArgumentException("limit may not be negative");
                }
            }
            this.limit = limit;
            return this;
        }
    }
}
