package com.atlassian.marketplace.client.api;

import com.atlassian.marketplace.client.HttpConfiguration;
import com.atlassian.marketplace.client.MpacException;
import com.atlassian.marketplace.client.model.Application;
import com.atlassian.marketplace.client.model.ApplicationSummary;
import com.atlassian.marketplace.client.model.ApplicationVersion;
import com.atlassian.upm.api.util.Option;

/**
 * Starting point for all resources that return plugin information.
 */
public interface Applications
{
    /**
     * Returns the list of all applications.
     * @return the list of all applications
     * @throws MpacException  if the server was unavailable or returned an error
     */
    Iterable<ApplicationSummary> findAll() throws MpacException;

    /**
     * Queries the details for a single application.
     * @param query  an {@link ApplicationDetailQuery}
     * @return  an {@link Application}, or {@link Option#none()} if the application key was not found
     * @throws MpacException  if the server was unavailable or returned an error
     */
    Option<Application> get(ApplicationDetailQuery query) throws MpacException;
    
    /**
     * Queries the details for a specific application version.  This is similar to calling
     * {@link #get(ApplicationDetailQuery)} with a query that specifies a single version, except that
     * 1. it does not include the rest of the application data, and 2. it can return an unpublished version,
     * whereas {@link #get(ApplicationDetailQuery)} will not include unpublished versions unless you have
     * administrator credentials.
     * @param application  the key of the application
     * @param buildNumber  the build number for the desired version
     * @return  an {@link ApplicationVersion}, or {@link Option#none()} if the version was not found
     * @throws MpacException  if the server was unavailable or returned an error
     */
    Option<ApplicationVersion> getVersion(ApplicationKey application, long buildNumber) throws MpacException;

    /**
     * Adds or updates version information for an application.  You can only do this if you have
     * provided administrator credentials in your {@link HttpConfiguration}.
     * <p>
     * The unique identifier of a version is the {@link ApplicationVersion#getBuildNumber() build number}: if you
     * provide an {@link ApplicationVersion} with the same build number as an existing version, the existing
     * information will be overwritten, otherwise a new version will be added.  The build number of an
     * existing version cannot be changed.
     * <p>
     * To construct the {@link ApplicationVersion} object, use {@link com.atlassian.marketplace.client.model.ModelBuilders#applicationVersion()}.
     * 
     * @param app  the application to update
     * @param version  the version data to add or update
     * @throws MpacException  if the server was unavailable or returned an error
     */
    void putVersion(ApplicationKey app, ApplicationVersion version) throws MpacException;
}
