package com.atlassian.marketplace.client.api;

import com.atlassian.upm.api.util.Option;

import static com.atlassian.upm.api.util.Option.none;
import static com.atlassian.upm.api.util.Option.some;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Encapsulates application search parameters that can be passed to {@link Plugins#get(ApplicationDetailQuery)}.
 */
public final class ApplicationDetailQuery
{
    public static final class BuildNumber
    {
        private final long buildNumber;
        private final boolean greaterThan;
        
        private BuildNumber(long buildNumber, boolean greaterThan)
        {
            this.buildNumber = buildNumber;
            this.greaterThan = greaterThan;
        }
        
        public static BuildNumber equalTo(long buildNumber)
        {
            return new BuildNumber(buildNumber, false);
        }
        
        public static BuildNumber greaterThan(long buildNumber)
        {
            return new BuildNumber(buildNumber, true);
        }
        
        public long getBuildNumber()
        {
            return buildNumber;
        }
        
        public boolean isGreaterThan()
        {
            return greaterThan;
        }
    }
    
    private final ApplicationKey appKey;
    private final Option<BuildNumber> buildNumber;
    private final Option<Integer> limitVersions;
    
    /**
     * Returns a new {@link Builder} for constructing an ApplicationDetailQuery.
     * @param appKey  the application key (required)
     */
    public static Builder builder(ApplicationKey appKey)
    {
        return new Builder(appKey);
    }
    
    private ApplicationDetailQuery(Builder builder)
    {
        appKey = builder.appKey;
        buildNumber = builder.buildNumber;
        limitVersions = builder.limitVersions;
    }
    
    public ApplicationKey getAppKey()
    {
        return appKey;
    }
    
    public Option<BuildNumber> getBuildNumber()
    {
        return buildNumber;
    }
    
    public Option<Integer> getLimitVersions()
    {
        return limitVersions;
    }
    
    public static class Builder
    {
        private final ApplicationKey appKey;
        private Option<BuildNumber> buildNumber = none();
        private Option<Integer> limitVersions = some(1);
        
        public Builder(ApplicationKey appKey)
        {
            this.appKey = checkNotNull(appKey, "appKey");
        }
        
        public ApplicationDetailQuery build()
        {
            return new ApplicationDetailQuery(this);
        }
        
        /**
         * Filters the result to only include a specified application version or version range.
         * @param buildNumber  a {@link BuildNumber} object indicating which version(s) to include, or
         *   {@link Option#none} for no build number filtering (i.e. return all versions)
         * @return  the same Builder
         */
        public Builder buildNumber(Option<BuildNumber> buildNumber)
        {
            this.buildNumber = checkNotNull(buildNumber);
            return this;
        }

        /**
         * Specifies whether to include details for all versions of the application or a subset thereof
         * in the {@link com.atlassian.marketplace.client.model.Application#getVersions} property of the
         * returned application. The default is to include only the most recent one.
         * @param limitVersions  {@link Option#some}(N) to return the most recent N versions;
         *   {@link Option#none} for as many versions as possible
         * @return  the same Builder
         */
        public Builder limitVersions(Option<Integer> limitVersions)
        {
            this.limitVersions = checkNotNull(limitVersions);
            return this;
        }
    }
}
