package com.atlassian.templaterenderer.plugins;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.event.PluginEventManager;
import com.atlassian.plugin.remotable.api.annotation.ComponentImport;
import com.atlassian.plugin.remotable.api.annotation.PublicComponent;
import com.atlassian.plugin.tracker.DefaultPluginModuleTracker;
import com.atlassian.plugin.tracker.PluginModuleTracker;
import com.atlassian.templaterenderer.TemplateContextFactory;

import java.util.Map;
import java.util.HashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;

/**
 * Implementation of the template context factory
 */
@Named
@Singleton
@PublicComponent(TemplateContextFactory.class)
public class TemplateContextFactoryImpl implements TemplateContextFactory
{
    private static final Logger log = LoggerFactory.getLogger(TemplateContextFactoryImpl.class);
    private final PluginModuleTracker<Object, TemplateContextItemModuleDescriptor> templateContextItemTracker;

    @Inject
    public TemplateContextFactoryImpl(
            @ComponentImport PluginAccessor pluginAccessor,
            @ComponentImport PluginEventManager eventManager)
    {
        this.templateContextItemTracker = new DefaultPluginModuleTracker<Object, TemplateContextItemModuleDescriptor>(
                pluginAccessor, eventManager, TemplateContextItemModuleDescriptor.class);
    }

    /**
     * Create a context for a template renderer
     *
     * @param contextParams Any extra context parameters that should be added to the context
     * @return A map of the context
     */
    public Map<String, Object> createContext(String pluginKey, Map<String, Object> contextParams)
    {
        Map<String, Object> context = new HashMap<String, Object>();
        context.put("context", context);

        for (TemplateContextItemModuleDescriptor desc : templateContextItemTracker.getModuleDescriptors())
        {
            if (desc.isGlobal() || desc.getPluginKey().equals(pluginKey))
            {
                try
                {
                    context.put(desc.getContextKey(), desc.getModule());
                }
                catch (RuntimeException re)
                {
                    log.error("Error loading module for " + desc.getPluginKey() + ":" + desc.getKey(), re);
                }
            }
        }
        context.putAll(contextParams);
        return context;
    }

    @PreDestroy
    public void destroy()
    {
        templateContextItemTracker.close();
    }
}
