package com.atlassian.support.tools.salext;

import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sisyphus.RemoteXmlPatternSource;
import com.atlassian.sisyphus.SisyphusPatternSource;
import com.atlassian.support.tools.properties.MultiValuePropertyStore;
import com.atlassian.support.tools.properties.PropertyStore;
import com.atlassian.support.tools.properties.SupportInfoAppenderManager;
import com.atlassian.support.tools.properties.SupportInfoXmlKeyResolver;
import com.atlassian.support.tools.salext.bundle.ApplicationInfoBundle;
import com.atlassian.support.tools.salext.bundle.ApplicationPropertiesInfoBundle;
import com.atlassian.support.tools.salext.bundle.BundleManifest;
import com.atlassian.support.tools.salext.bundle.ThreadDumpBundle;
import com.atlassian.support.tools.salext.output.XmlSupportInfoFormatter;
import com.atlassian.templaterenderer.TemplateRenderer;
import com.opensymphony.module.sitemesh.util.Container;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang.StringUtils;
import org.osgi.framework.BundleReference;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.URL;
import java.util.*;
import java.util.regex.Pattern;


public abstract class AbstractSupportApplicationInfo implements SupportApplicationInfo {

	protected static final String ENABLED_PLUGINS = "stp.properties.plugins.enabled";

	private boolean isTomcat;
	protected final ApplicationProperties applicationProperties;

	protected final I18nResolver i18nResolver;

	protected final List<ApplicationInfoBundle> applicationInfoBundles = new ArrayList<ApplicationInfoBundle>();
	protected ServletContext servletContext;
	protected final UserManager userManager;
    protected final TemplateRenderer renderer;
    protected final SupportInfoAppenderManager supportInfoAppenderManager;
    protected final SupportInfoXmlKeyResolver supportInfoXmlKeyResolver;

	protected final Map<String, Map<String, String>> propertiesByCategory = new LinkedHashMap<String, Map<String, String>>();

	public static final Map<String, List<Pattern>> FILE_PATTERNS = new HashMap<String, List<Pattern>>();
	private static final Pattern TOMCAT_USERS_SANITIZER_PATTERN = Pattern.compile(
			"(?:.*(?:username|password|name)[ ]*=[ ]*[\"']?([^\"'> ]*)[\"']?.*)", Pattern.CASE_INSENSITIVE);

	static {
		FILE_PATTERNS
				.put("server.xml",
						Arrays.asList(Pattern
								.compile(
										"(?:.*(?:username|password|keystorePass|truststorePass|connectionPassword|connectionName)[ ]*=[ ]*[\"']?([^\"'> ]*)[\"']?.*)",
										Pattern.CASE_INSENSITIVE)));

		FILE_PATTERNS
				.put("crowd.properties", Arrays.asList(Pattern.compile("application\\.(?:name|password)\\s+(.+)\\s*",
						Pattern.CASE_INSENSITIVE)));

		FILE_PATTERNS.put("tomcat-users.xml", Arrays.asList(TOMCAT_USERS_SANITIZER_PATTERN));
		
		FILE_PATTERNS.put("atlassian-user.xml", Arrays.asList( 
				Pattern.compile("(?:securityPrincipal\\s*>)(.*)(?:</\\s*securityPrincipal)"), 
				Pattern.compile("(?:securityCredential\\s*>)(.*)(?:</\\s*securityCredential)"))); 

	}

    public AbstractSupportApplicationInfo(ApplicationProperties applicationProperties,
                                          I18nResolver i18nResolver,
                                          UserManager userManager,
                                          TemplateRenderer renderer,
                                          SupportInfoAppenderManager supportInfoAppenderManager,
                                          SupportInfoXmlKeyResolver supportInfoXmlKeyResolver)
    {
        this.applicationProperties = applicationProperties;
        this.i18nResolver = i18nResolver;
        this.userManager = userManager;
        this.renderer = renderer;
        this.supportInfoAppenderManager = supportInfoAppenderManager;
        this.supportInfoXmlKeyResolver = supportInfoXmlKeyResolver;

        this.applicationInfoBundles.add(new ApplicationPropertiesInfoBundle(BundleManifest.APPLICATION_PROPERTIES,
                "stp.zip.include.application.properties", "stp.zip.include.application.properties.description", this));
        this.applicationInfoBundles.add(new ThreadDumpBundle(BundleManifest.THREAD_DUMP, "stp.zip.include.threadDump",
                "stp.zip.include.threadDump.description", null, this));
    }

	protected AbstractSupportApplicationInfo() {
		this.applicationProperties = null;
		this.i18nResolver = null;
		this.userManager = null;
		this.renderer = null;
		this.supportInfoAppenderManager = null;
		this.supportInfoXmlKeyResolver = null;
	}

	@Override
    public String getUserName() {
        try {
            return this.userManager.getRemoteUsername();
        } catch (java.lang.Exception exception) {
            return "";  //To change body of catch statement use File | Settings | File Templates.
        }
    }

    @Override
    public String getUserEmail() {
        try {
            return this.userManager.getUserProfile(getUserName()).getEmail();
        } catch (java.lang.Exception exception) {
            return "";
        }
    }

	@Override
	public final List<ApplicationInfoBundle> getApplicationFileBundles() {
		return this.applicationInfoBundles;
	}

	@Override
	public void initServletInfo(ServletConfig config) {
		this.servletContext = config.getServletContext();
		this.isTomcat = config.getServletContext().getServerInfo().contains("Tomcat");
	}

	@Override
	public PropertyStore loadProperties() {
		PropertyStore store = new MultiValuePropertyStore();

        supportInfoAppenderManager.addSupportInfo(store);

		return store;
	}

	@Override
	public boolean isTomcat() {
		return this.isTomcat;
	}

	protected SisyphusPatternSource getPatternSourceByURL(String urlString, Proxy proxy) throws IOException, ClassNotFoundException {
		return new RemoteXmlPatternSource(new URL(urlString), proxy);
	}

	public String findTomcatFileOrDirectory(String fileOrDirectoryName) {
		String catalinaBase = System.getProperty("catalina.base");
		File file = new File(catalinaBase, fileOrDirectoryName);
		if (file.exists()) {
			return file.getAbsolutePath();

		}

		String catalinaHome = System.getProperty("catalina.home");
		file = new File(catalinaHome, fileOrDirectoryName);
		if (file.exists()) {
			return file.getAbsolutePath();
		}

		String workingDirectory = System.getProperty("working.dir");
		file = new File(workingDirectory + "../", fileOrDirectoryName);
		if (file.exists()) {
			return file.getAbsolutePath();
		}

		return null;
	}

	@Override
	public String getApplicationName() {
		return this.applicationProperties.getDisplayName();
	}

	@Override
	public String getApplicationVersion() {
		return this.applicationProperties.getVersion();
	}

	@Override
	public String getApplicationBuildNumber() {
		return this.applicationProperties.getBuildNumber();
	}

	@Override
	public Date getApplicationBuildDate() {
		return this.applicationProperties.getBuildDate();
	}

	@Override
	public String getApplicationHome() {
		return this.applicationProperties.getHomeDirectory().toString();
	}

	@Override
	public String getText(String key) {
		return this.i18nResolver.getText(key);
	}

	@Override
	public String getText(String key, Serializable... arguments) {
		return this.i18nResolver.getText(key, arguments);
	}

	@Override
	public String getServletContextPath(String pathToLookup) {
		return this.servletContext.getRealPath(pathToLookup);
	}

	@Override
	public void flagSelectedApplicationFileBundles(HttpServletRequest req) {
		List<ApplicationInfoBundle> selectedApplicationFileBundles = getSelectedApplicationInfoBundles(req);

		// We do this in the inverse so that the default is true. This supports
		// "check by default" but still preserves "unticked" boxes if we display
		// warnings.
		for (ApplicationInfoBundle bundle : this.applicationInfoBundles) {
			if (selectedApplicationFileBundles != null && selectedApplicationFileBundles.size() > 0) {
				if (!selectedApplicationFileBundles.contains(bundle)) {
					bundle.setSelected(false);
				}
			} else {
				bundle.setSelected(true);
			}
		}
	}

	@Override
	public List<ApplicationInfoBundle> getSelectedApplicationInfoBundles(HttpServletRequest req) {
		List<ApplicationInfoBundle> selectedApplicationInfoBundles = new ArrayList<ApplicationInfoBundle>();
		for (ApplicationInfoBundle applicationInfoBundle : this.applicationInfoBundles) {
			final String flagValue = req.getParameter(applicationInfoBundle.getKey());
			if (flagValue != null && Boolean.parseBoolean(flagValue)) {
				selectedApplicationInfoBundles.add(applicationInfoBundle);
			}
		}

		return selectedApplicationInfoBundles;
	}

	@Override
	public Map<String, String> getPropertiesByCategory(String category) {
		return this.propertiesByCategory.get(category);
	}

	@Override
	public Set<String> getPropertyCategories() {
		return this.propertiesByCategory.keySet();
	}

	public String saveProperties() {
		PropertyStore supportInfoProperties = loadProperties();
        Properties xmlElementNameMappings = supportInfoXmlKeyResolver.getKeyMappings();
        return XmlSupportInfoFormatter.getFormattedProperties(supportInfoProperties, xmlElementNameMappings, getStpVersion(), getApplicationName(), getApplicationVersion(), getApplicationSEN(), getApplicationServerID());
    }

	private String getStpVersion() {
		return ((BundleReference) getClass().getClassLoader()).getBundle().getVersion().toString();
	}

	@Override
	public TemplateRenderer getTemplateRenderer() {
		return this.renderer;
	}

	@Override
	public String getBaseURL(HttpServletRequest req) {
		return req.getRequestURI().replaceFirst(req.getServletPath() + ".*", "");
	}

	@Override
	public List<String> getSystemWarnings() {
		return Collections.emptyList();
	}

	/**
	 * @return a readable version of the current container, or "Unknown".
	 */
	public String getAppServer() {
		switch (Container.get()) {
		case Container.TOMCAT:
			return "Apache Tomcat";
		case Container.ORION:
			return "Orion";
		case Container.WEBLOGIC:
			return "IBM WebLogic";
		case Container.JRUN:
			return "JRUN";
		case Container.RESIN:
			return "RESIN" + Container.get();
		case Container.HPAS:
			return "HPAS";
		case Container.UNKNOWN:
			return "Unknown";
		}
		return "Unknown";
	}

	@Override
	public String getExportDirectory() {
		return getApplicationHome() + "/export";
	}

	@Override
	public String getFromAddress() {
		return "noreply@atlassian.com";
	}
	
	/**
	 * Get the proxy from the system parameters
	 * @return The proxy
	 */
	protected Proxy getProxy() {
		// Check for an http proxy 
		String proxy = System.getProperty("http.proxyHost");
		if (StringUtils.isNotEmpty(proxy))
		{
			// if one exists, return it with the port, or 80 if no port is specified
			return new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxy, Integer.getInteger("http.proxyPort", 80)));
		}
		else
		{
			// Check for an https proxy
			proxy = System.getProperty("https.proxyHost");
			
			if (StringUtils.isNotEmpty(proxy))
			{
				// if one exists, return it with the port, or 443 if no port is specified
				return new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxy, Integer.getInteger("https.proxyPort", 443)));
			}
		}
			
		return Proxy.NO_PROXY;
	}
}
