package com.atlassian.support.tools.rest;

import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.support.tools.salext.SupportApplicationInfo;
import com.atlassian.support.tools.salext.license.ApplicationLicenseInfo;
import org.apache.commons.lang.StringUtils;

import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.CacheControl;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.Date;

@Produces({MediaType.APPLICATION_JSON})
@Path("license")
public class LicenseStatusService
{
    private final static int DAY_IN_MS = 24 * 60 * 60 * 1000;
    private final UserManager userManager;
    private final PluginSettings pluginSettings;
    private final SupportApplicationInfo appInfo;
    private static final String BANNER_HIDDEN_TILL_DAY = "bannerHiddenTillDay";

    public LicenseStatusService(final UserManager userManager, final PluginSettingsFactory pluginSettingsFactory, final SupportApplicationInfo appInfo)
    {
        // Need to create global settings, because FeCru doesn't like "key" settings - they have to correlate with repo's
        pluginSettings = pluginSettingsFactory.createGlobalSettings();
        this.userManager = userManager;
        this.appInfo = appInfo;
    }

    @GET
    @Path("/status")
    @Produces(MediaType.APPLICATION_JSON)
    public Response status()
    {
        String remoteUsername = userManager.getRemoteUsername();
        if (StringUtils.isBlank(remoteUsername))
        {
            return Response.status(401).cacheControl(never()).build();
        }

        if (!userManager.isAdmin(remoteUsername))
        {
            return Response.status(403).cacheControl(never()).build();
        }

        ApplicationLicenseInfo licenseInfo = appInfo.getLicenseInfo();
        if (licenseInfo.isEvaluation())
        {
            return Response.status(204).build();
        }

        int daysToExpiry = getDaysToExpiry(licenseInfo.getMaintenanceExpiryDate());

        if (daysToExpiry > 30 || daysToExpiry < 0)
        {
            resetBannerHiddenProperty(remoteUsername);
            return Response.status(204).build();
        }

        int hiddenTillDays = getBannerHiddenTillDays(remoteUsername);
        if (shouldKeepBannerHiddenBanner(hiddenTillDays, daysToExpiry))
        {
            return Response.status(204).build();
        }

        return Response.ok(new LicenseStatus(appInfo.getApplicationName(), appInfo.getApplicationSEN(), daysToExpiry)).build();
    }

    @POST
    @Path("/remindMeLater")
    public Response remindMeLater()
    {
        String remoteUsername = userManager.getRemoteUsername();
        if (StringUtils.isBlank(remoteUsername) || !userManager.isAdmin(remoteUsername))
        {
            return Response.status(401).cacheControl(never()).build();
        }

        int daysToExpiry = getDaysToExpiry(appInfo.getLicenseInfo().getMaintenanceExpiryDate());

        if (daysToExpiry > 30 || daysToExpiry < 0)
        {
            resetBannerHiddenProperty(remoteUsername);
        }
        else
        {
            int hiddenTillDays = daysToExpiry / 2;
            if (hiddenTillDays > 7)
            {
                hiddenTillDays = 15;
            }
            else if (hiddenTillDays > 0)
            {
                hiddenTillDays = 7;
            }
            else
            {
                hiddenTillDays = 0;
            }

            pluginSettings.put(getUserSettingKey(remoteUsername), ((Integer) hiddenTillDays).toString());
        }
        return Response.ok().build();
    }

    @POST
    @Path("/remindMeNever")
    public Response remindMeN()
    {
        String remoteUsername = userManager.getRemoteUsername();
        if (StringUtils.isBlank(remoteUsername) || !userManager.isAdmin(remoteUsername))
        {
            return Response.status(401).cacheControl(never()).build();
        }

        int daysToExpiry = getDaysToExpiry(appInfo.getLicenseInfo().getMaintenanceExpiryDate());

        if (daysToExpiry > 30 || daysToExpiry < 0)
        {
            resetBannerHiddenProperty(remoteUsername);
        }
        else
        {
            pluginSettings.put(getUserSettingKey(remoteUsername), "-1");
        }
        return Response.ok().build();
    }

    @POST
    @Path("/clearReminders")
    public Response clearReminders()
    {
        String remoteUsername = userManager.getRemoteUsername();
        if (StringUtils.isBlank(remoteUsername) || !userManager.isAdmin(remoteUsername))
        {
            return Response.status(401).cacheControl(never()).build();
        }

        resetBannerHiddenProperty(remoteUsername);
        return Response.ok().build();
    }

    private boolean shouldKeepBannerHiddenBanner(final int hiddenTillDays, final int daysToExpiry)
    {
        // every time they click hide the banner - we will halve the current number of days, round it to either 15, 7 or 0 and save that number
        // to decide whether to keep the banner hidden we simply check if current days to expiry is greater than the saved number of days or not.
        // if it is greater - keep it hidden
        // if it is less or equal - show the banner
        // except if the user never wanted to be reminded again for this license, in which case the limit is set to -1

        if (hiddenTillDays == -1)
        {
            return true;
        }

        if (hiddenTillDays == 0)
        {
            return daysToExpiry < 7; // only on the 7th day we will start allowing to hide this forever.
        }

        return daysToExpiry > hiddenTillDays;
    }

    private void resetBannerHiddenProperty(final String username)
    {
        pluginSettings.remove(getUserSettingKey(username));
    }

    private int getBannerHiddenTillDays(final String username)
    {
        String settingValue = (String) pluginSettings.get(getUserSettingKey(username));
        if (StringUtils.isBlank(settingValue))
        {
            return 30;
        }
        try
        {
            return Integer.parseInt(settingValue);
        }
        catch (NumberFormatException e)
        {
            return 30;
        }
    }

    private int getDaysToExpiry(Date expiryDate)
    {
        int daysToExpiry = 365;
        if (expiryDate != null)
        {
            daysToExpiry = (int) ((expiryDate.getTime() - System.currentTimeMillis()) / DAY_IN_MS);
        }
        return daysToExpiry;
    }

    private static CacheControl never()
    {
        CacheControl cacheNever = new javax.ws.rs.core.CacheControl();
        cacheNever.setNoStore(true);
        cacheNever.setNoCache(true);
        return cacheNever;
    }

    private String getUserSettingKey(String username)
    {
        return BANNER_HIDDEN_TILL_DAY + username;
    }
}
