package com.atlassian.support.tools.properties.appenders;

import com.atlassian.support.tools.spi.RootLevelSupportInfoAppender;
import com.atlassian.support.tools.spi.SupportInfoBuilder;
import com.opensymphony.module.sitemesh.util.Container;
import org.apache.commons.lang.StringUtils;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.RuntimeMXBean;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;

import static com.atlassian.support.tools.salext.SupportApplicationInfo.*;

public class JavaSupportInfoAppender extends RootLevelSupportInfoAppender
{

    private static final DecimalFormat PERCENTAGE_FORMAT = new DecimalFormat("###%");

    private final RuntimeMXBean runtimeMXBean;
    private final MemoryPoolMXBean permGenMXBean;

    public JavaSupportInfoAppender()
    {
        runtimeMXBean = ManagementFactory.getRuntimeMXBean();

        MemoryPoolMXBean permGenMXBean = null;
        // get the heap and permgen MemoryPoolMXBean objects
        for (MemoryPoolMXBean bean : ManagementFactory.getMemoryPoolMXBeans())
        {
            if (bean.getName().contains("Perm Gen"))
            {
                permGenMXBean = bean;
                break;
            }
        }
        this.permGenMXBean = permGenMXBean;
    }

    @Override
    protected void addSupportInfo(SupportInfoBuilder builder)
    {
        builder = builder.addCategory("stp.properties.java");
        Properties systemProps = System.getProperties();
        for (Map.Entry<Object, Object> entry : systemProps.entrySet())
        {
            builder.addValue(entry.getKey().toString(), entry.getValue().toString());
        }

        builder.addValue(JAVA_VM_ARGUMENTS, getJVMInputArguments());
        builder.addValue(APPLICATION_SERVER, getAppServer());

        // Memory properties
        builder.addValue(JAVA_HEAP_USED, getTotalHeap());
        builder.addValue(JAVA_HEAP_AVAILABLE, getFreeHeap());
        builder.addValue(JAVA_HEAP_PERCENT_USED, getPercentageHeapUsed());
        builder.addValue(JAVA_HEAP_MAX, getMaxHeap());

        builder.addValue(JAVA_PERMGEN_USED, getPermgenUsed());
        builder.addValue(JAVA_PERMGEN_MAX, getMaxPermgen());
    }

    private String getFormattedNum(long num)
    {
        NumberFormat nf = NumberFormat.getInstance(Locale.ENGLISH);
        return nf.format(num);
    }

    private String getJVMInputArguments()
    {
        if (this.runtimeMXBean != null)
        {
            return StringUtils.join(this.runtimeMXBean.getInputArguments(), " ");
        }
        return "Unknown";
    }

    /**
     * @return a readable version of the current container, or "Unknown".
     */
    public String getAppServer()
    {
        switch (Container.get())
        {
            case Container.TOMCAT:
                return "Apache Tomcat";
            case Container.ORION:
                return "Orion";
            case Container.WEBLOGIC:
                return "IBM WebLogic";
            case Container.JRUN:
                return "JRUN";
            case Container.RESIN:
                return "RESIN" + Container.get();
            case Container.HPAS:
                return "HPAS";
            case Container.UNKNOWN:
                return "Unknown";
            default:
                return "Unknown";
        }
    }

    private String getMaxHeap()
    {
        return getFormattedNum(Runtime.getRuntime().maxMemory());
    }

    private String getTotalHeap()
    {
        return getFormattedNum(Runtime.getRuntime().totalMemory());
    }

    private String getFreeHeap()
    {
        return getFormattedNum(Runtime.getRuntime().freeMemory());
    }

    private String getPercentageHeapUsed()
    {
        long total = Runtime.getRuntime().totalMemory();
        long free = Runtime.getRuntime().freeMemory();
        double percent = total != 0 ? (double) (total - free) / total : 1;
        return PERCENTAGE_FORMAT.format(percent);
    }

    private String getPermgenUsed()
    {
        if (this.permGenMXBean != null)
        {
            return getFormattedNum(this.permGenMXBean.getUsage().getUsed());
        }
        return "Unknown";
    }

    private String getMaxPermgen()
    {
        if (this.permGenMXBean != null)
        {
            return getFormattedNum(this.permGenMXBean.getUsage().getMax());
        }
        return "Unknown";
    }
}
