package com.atlassian.support.tools.properties;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.Resources;
import com.atlassian.plugin.elements.ResourceDescriptor;
import com.atlassian.plugin.event.PluginEventListener;
import com.atlassian.plugin.event.PluginEventManager;
import com.atlassian.plugin.event.events.PluginDisabledEvent;
import com.atlassian.plugin.event.events.PluginEnabledEvent;
import com.atlassian.util.concurrent.CopyOnWriteMap;
import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;

public class PluginSupportInfoXmlKeyResolver implements SupportInfoXmlKeyResolver
{

    private static final String SUPPORT_INFO_XML = "support-info-xml";
    private static final Logger log = Logger.getLogger(PluginSupportInfoXmlKeyResolver.class);

    private final Map<Plugin, Iterable<String>> xmlNamesResources = CopyOnWriteMap.<Plugin, Iterable<String>>builder().newHashMap();

    public PluginSupportInfoXmlKeyResolver(PluginAccessor pluginAccessor, PluginEventManager pluginEventManager)
    {
        pluginEventManager.register(this);
        resolveResources(pluginAccessor.getEnabledPlugins());
    }

    @Override
    public Properties getKeyMappings()
    {
        Properties xmlNames = new Properties();

        for (Map.Entry<Plugin, Iterable<String>> pluginResources : xmlNamesResources.entrySet())
        {
            for (String resource : pluginResources.getValue())
            {
                InputStream stream = pluginResources.getKey().getClassLoader().getResourceAsStream(resource);
                if (stream == null)
                {
                    log.error("Error loading xml elements from resource file.");
                } else
                {
                    try
                    {
                        xmlNames.load(stream);
                    } catch (IOException e)
                    {
                        log.error("Error loading xml element names for use in application property files:", e);
                    } finally
                    {
                        IOUtils.closeQuietly(stream);
                    }
                }
            }
        }

        return xmlNames;
    }

    @PluginEventListener
    public void onPluginDisabled(PluginDisabledEvent event)
    {
        xmlNamesResources.remove(event.getPlugin());
    }

    @PluginEventListener
    public void onPluginEnabled(PluginEnabledEvent event)
    {
        Plugin plugin = event.getPlugin();
        resolveResources(Collections.singleton(plugin));
    }

    private void resolveResources(Collection<Plugin> plugins)
    {
        for (Plugin plugin : plugins)
        {
            Collection<ResourceDescriptor> descriptors =
                    Collections2.filter(plugin.getResourceDescriptors(), new Resources.TypeFilter(SUPPORT_INFO_XML));

            if (!descriptors.isEmpty())
            {
                Set<String> locations = ImmutableSet.copyOf(Collections2.transform(descriptors,
                        new Function<ResourceDescriptor, String>()
                        {
                            public String apply(ResourceDescriptor descriptor)
                            {
                                return descriptor.getLocation();
                            }
                        }));
                xmlNamesResources.put(plugin, locations);
            }
        }
    }

}
