package com.atlassian.support.tools.properties;

import org.apache.commons.lang.StringEscapeUtils;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.NotThreadSafe;
import java.util.*;
import java.util.Map.Entry;

/**
 * Stores the support information in a hierarchical properties format
 * <p/>
 * This implementation is <em>not</em> thread safe.
 */
@NotThreadSafe
public class MultiValuePropertyStore implements PropertyStore
{
    private final Map<String, List<PropertyStore>> categories = new LinkedHashMap<String, List<PropertyStore>>();
    private final Map<String, String> values = new LinkedHashMap<String, String>();

    @Nonnull
    @Override
    public void setValue(@Nonnull String name, String value)
    {
        values.put(escapeXmlElementName(name), StringEscapeUtils.escapeXml(value));
    }

    @Nonnull
    @Override
    public Map<String, String> getValues()
    {
        return Collections.unmodifiableMap(values);
    }

    @Override
    public void putValues(@Nonnull Map<String, String> newValues)
    {
        for (Map.Entry<String, String> entry : newValues.entrySet())
        {
            setValue(entry.getKey(), entry.getValue());
        }
    }

    @Nonnull
    @Override
    public PropertyStore addCategory(@Nonnull String categoryKey)
    {
        return addCategory(categoryKey, new MultiValuePropertyStore());
    }

    @Nonnull
    @Override
    public PropertyStore addCategory(@Nonnull String categoryKey, @Nonnull PropertyStore store)
    {
        List<PropertyStore> propertyStores = categories.get(categoryKey);
        ArrayList<PropertyStore> localCopy = new ArrayList<PropertyStore>();
        if (propertyStores != null)
        {
            localCopy.addAll(propertyStores);
        }
        localCopy.add(store);
        categories.put(categoryKey, localCopy);
        return store;
    }

    @Nonnull
    @Override
    public Map<String, List<PropertyStore>> getCategories()
    {
        // return a deep copy of the categories
        Map<String, List<PropertyStore>> transformedCategories = new LinkedHashMap<String, List<PropertyStore>>();
        for (Entry<String, List<PropertyStore>> entry : categories.entrySet())
        {
            transformedCategories.put(entry.getKey(), new ArrayList<PropertyStore>(entry.getValue()));
        }
        return transformedCategories;
    }

    /**
     * Escape a key that will be used as an XML element name, replacing colons with dashes to avoid
     * "this:that" being interpreted as element "that" in namespace "this".
     *
     * @param key intended for use as an XML element name.
     * @return The escaped value of the key.
     */
    private String escapeXmlElementName(String key)
    {
        String escapedKey = StringEscapeUtils.escapeXml(key);
        return escapedKey.replace(":", "-");
    }
}
