/**
 * Provides a way to register file-handlers to handle rendering the source of a file or the diff of file changed.
 * For example, a {@linkCode FileHandler} can be registered to handle .stl files and render them as 3D files.
 * JS files registering file-handlers should use the resource context 'stash.feature.files.fileHandlers'.
 *
 * **This module is available synchronously.**
 *
 * **Web Resource:** com.atlassian.stash.stash-web-api:file-handlers
 *
 * @namespace stash/api/feature/files/file-handlers
 *
 * @example
 * function MyView(options) {
 *     var $element = $('&lt;div/&gt;');
 *     $container.append($element);
 *
 *     return {
 *         destroy: function() {
 *             $element.remove();
 *         },
 *         extraClasses: 'my-class something-else'
 *     };
 * }
 *
 * var myFileHandler = {
 *     weight: 400,
 *     handle: function(options) {
 *         if (options.fileChange.extension === 'stl') {
 *             return new MyView(options);
 *         } else if (options.fileChange.extension === 'stl2') {
 *             // Returning a rejected promise with a message will display errors appropriately
 *             return $.Deferred().reject('File extension not supported');
 *         }
 *         // Return null/undefined or an empty promise to pass silently
 *     }
 * };
 *
 * // Register your handler immediately to ensure it is in place when needed.
 * // Note that we use the synchronous syntax to require the module, which ensures there is no delay before
 * // register() is called.
 * require('feature/file-content/file-handlers').register(myFileHandler);
 */
define('stash/api/feature/files/file-handlers', [
    'util/handler-registry'
],
/**
 * @exports stash/api/feature/files/file-handlers
 */
function(
    HandlerRegistry
) {

    'use strict';

    /**
     * @typedef {Object}    FileHandlingContext
     * @memberOf stash/api/feature/files/file-handlers
     *
     * @property {string}           contentMode - The mode of content. This is either 'source' or 'diff'.
     * @property {jQuery}           $container  - A jQuery node in which you should append your rendered file content.
     * @property {JSON.FileChangeJSON}   fileChange - Describes the changed file.
     * @property {string}           commentMode - The comment rendering mode. "none", "read", "reply-only", or "create-new"
     * @property {boolean}          isExcerpt - Indicates whether this is only an excerpt and not a full file/diff.
     * @property {number}           targetLine - The line number of the anchored line.
     * @property {Function}         [diffUrlBuilder] - An optional function that accepts a {JSON.FileChangeJSON} and will return
     *                                                 a {stash/api/util/navbuilder.Builder} to the built-in Stash REST
     *                                                 endpoint to obtain the diff information.
     * @property {Array<JSON.CommentJSON>}    [lineComments] - An array of comments anchored to the lines of the file. The structure
     *                                      matches the structure for comments retrieved via the REST API. This is only provided
     *                                      for handling diffs as activity items for a pull request. For other usages, line comments
     *                                      will be retrieved from the server when retrieving diffs.
     */

    /**
     * Callback to handle file.
     *
     * @callback FileHandleCallback
     * @memberOf stash/api/feature/files/file-handlers
     *
     * @param {stash/api/feature/files/file-handlers.FileHandlingContext}   context - A map of properties describing the content
     *                                                              to be rendered and the context in which it is to be rendered.
     * @return {Promise}        A promise object that resolves with {@linkcode FileHandlerResult} if this handler will handle the request, or rejects otherwise.
     */

    /**
     * An object that can either render file sources, file diffs, or both for a particular subset of files. 
     *
     * @typedef {Object}    FileHandler
     * @memberOf stash/api/feature/files/file-handlers
     *
     * @property {number}               [weight=1000] - The weight of handler determining the order it is tried.
     *                                                  The default weight of the source/diff view is 1000.
     * @property {stash/api/feature/files/file-handlers.FileHandleCallback}   handle        - A function called to handle file content rendering.
     */

    /**
     * @typedef {Object}    FileHandlerResult
     * @memberOf stash/api/feature/files/file-handlers
     *
     * @property {Function} destroy        - A function that will be called before the current view is destroyed, which may happen on state change.
     *                                       This is a chance to destroy/cleanup any event listeners and remove DOM elements.
     * @property {string}   [extraClasses] - Additional style class applied to parent file-content. Can be used to apply background color.
     */

    // At the moment file-handlers is just a single, global HandlerRegistry
    return new HandlerRegistry();

});
