define('feature/pull-request/pull-request-create', [
    'bacon',
    'jquery',
    'underscore',
    'stash/api/util/navbuilder',
    'stash/api/util/server',
    'util/events',
    'util/function',
    'util/text',
    'model/page-state',
    'widget/markup-editor',
    'feature/pull-request/metadata-generator',
    'feature/user/user-multi-selector',
    'exports'
], function(
    Bacon,
    $,
    _,
    nav,
    server,
    events,
    fn,
    textUtil,
    pageState,
    MarkupEditor,
    metadataGenerator,
    UserMultiSelector,
    exports
    ) {

    "use strict";

    /**
     * Initialises the PR form
     *
     * @param parent               - A container to put the form in
     * @param submittedReviewers   - If this is a failed submission, the reviewers who should go in the box
     * @param {Bacon.Property<SourceTargetSelectorState>} selectorProperty - The current state of the branch selectors.
     * @param {Bacon.Property<string>} tabProperty - The current state of the branch selectors.
     * @param canCreatePRProperty - A stream of True/False events describing the refs in the sourceTargetSelector
     */
    function initPullRequestForm(parent, submittedReviewers, selectorProperty, tabProperty, canCreatePRProperty) {
        var destroyables = [];

        var $form = $(parent);

        MarkupEditor.bindTo($form.find('.markup-editor'));

        var currentUser = pageState.getCurrentUser();
        new UserMultiSelector($form.find('#reviewers'), {
            initialItems: submittedReviewers,
            excludedItems: currentUser ? [currentUser.toJSON()] : []
        });

        var $button = $form.find('#submit-form');

        var destroy = canCreatePRProperty
            .onValue(function(notEqual) {
                $button.enable(notEqual).attr('aria-disabled', !notEqual);
            });
        destroyables.push({destroy: destroy});

        var $title = $form.find("#title");
        if ($title.val() === '') {
            destroy = selectorProperty
                .map(_.compose(
                    textUtil.convertBranchNameToSentence.bind(textUtil),
                    fn.dotX('source.getDisplayId')
                ))
                .takeUntil($title.asEventStream('change'))
                .onValue($title.val.bind($title));

            destroyables.push({destroy: destroy});
        }

        if ($('#pull-request-description').val() === '') {
            destroyables.push({destroy: initDescriptionGeneration(selectorProperty, tabProperty)});
        }

        destroyables.push({destroy: initPageState(selectorProperty)});

        return {
            destroy: function() {
                _.invoke(destroyables, 'destroy');
            }
        };
    }

    /**
     * Initialise the PR create form description generation.
     *
     * @param {Bacon.Property<SourceTargetSelectorState>} selectorProperty - The current state of the branch selectors.
     * @param {Bacon.Property<string>} tabProperty - The current state of the branch selectors.
     * @returns {Function} to destroy all the listeners this function setup
     */
    function initDescriptionGeneration(selectorProperty, tabProperty) {
        var allSelected = selectorProperty.filter(function(state) {
            return state.source && state.target;
        });

        var waitingForCommitsTab = false;
        var pendingXHR;

        var onContentAdded = function(data) {
            if (waitingForCommitsTab) {
                waitingForCommitsTab = false;
                setDescription(metadataGenerator.generateDescriptionFromCommitsTableRows(data.values));
            }
        };

        var onRestDone = function(data) {
            setDescription(metadataGenerator.generateDescriptionFromCommitsJson(data));
            pendingXHR = null;
        };

        events.on('stash.widget.commitsTable.contentAdded', onContentAdded);

        var descriptionChangedStream = $('#pull-request-description')
            .asEventStream('keydown')
            .doAction(function(e) {
                $(e.target).data('description-changed', true);
                if (pendingXHR) {
                    pendingXHR.abort();
                    pendingXHR = null;
                }
            });

        var unsubSelector = Bacon.combineAsArray(allSelected, tabProperty)
            // combineAsArray seemed to produce duplicates that need to be skipped when changing tabs
            .skipDuplicates(function(a, b) {
                return a[0] === b[0] && a[1] === b[1];
            })
            .takeUntil(descriptionChangedStream)
            .slidingWindow(2, 1)
            .map(function(states) {
                // add a third item to the state indicating if the tab states are the same as previous
                if (states.length === 1) {
                    states[0].push(false);
                    return states[0];
                }
                states[1].push(states[0][1] !== states[1][1]);
                return states[1];
            })
            .onValue(function(states) {
                var selector = states[0];
                var tab = states[1];
                var tabStateChanged = states[2];

                if ((tabStateChanged && waitingForCommitsTab) ||
                    !tabStateChanged && tab !== 'commits') {
                    waitingForCommitsTab = false;
                    if (pendingXHR) {
                        pendingXHR.abort();
                        pendingXHR = null;
                    }
                    pendingXHR = updateDescriptionFromRest(selector.source, selector.target, onRestDone);
                } else if(!tabStateChanged && tab === 'commits') {
                    waitingForCommitsTab = true;
                }
            });


        return function() {
            events.off('stash.widget.commitsTable.contentAdded', onContentAdded);
            unsubSelector();
        };
    }

    /**
     * Makes a REST request to get the commit information to load into the description
     *
     * @param {object} source The source branch
     * @param {object} target The target branch
     * @param {Function} onRestDone A callback to call when the rest request returns
     * @returns {jqXHR}
     */
    function updateDescriptionFromRest(source, target, onRestDone) {
        console.log("Rest for: " + source.getDisplayId() + " " + target.getDisplayId());
        var url = nav.project(source.getRepository().getProject()).repo(source.getRepository()).commits().withParams({
            until : source.getLatestChangeset(),
            since :  target.getLatestChangeset(),
            secondaryRepositoryId : target.getRepository().getId(),
            start : 0,
            limit : 10
        }).build();

        return server.rest({
            type: 'GET',
            url: url,
            statusCode: { '*' : false } // fail silently.
        }).done(onRestDone);
    }

    /**
     * Sets the description of the Pull Request.
     *
     * @param description
     */
    function setDescription(description) {
        var $description = $('#pull-request-description');
        if(!$description.data('description-changed')) {
            $description.val(description).trigger('input');
        }
    }

    /**
     * Depends on sourceTargetSelector already having been initialised
     *
     * @param {Bacon.Property<SourceTargetSelectorState>} selectorProperty - The current state of the branch selectors.
     */
    function initPageState(selectorProperty) {

        pageState.extend('sourceRepository');
        pageState.extend('targetBranch');
        pageState.extend('sourceBranch');

        return selectorProperty.onValue(function(state) {
            pageState.setProject(state.targetRepo.getProject());

            pageState.setRepository(state.targetRepo);
            pageState.setSourceRepository(state.sourceRepo);

            pageState.setTargetBranch(state.target);
            pageState.setSourceBranch(state.source);
        });
    }

    exports.init = initPullRequestForm;
});