define('feature/file-content/side-by-side-diff-view', [
    'jquery',
    'underscore',
    'util/function',
    'util/math',
    'util/performance',
    'util/promise',
    'util/svg',
    'model/direction',
    'feature/file-content/diff-hunkmap',
    'feature/file-content/diff-view',
    'feature/file-content/diff-view-file-types',
    'feature/file-content/diff-view-segment-types',
    'feature/file-content/line-handle',
    'feature/file-content/side-by-side-diff-view/synchronized-scroll'
],
/**
 * Implement the Side-by-side Diff view for diffs.
 *
 * We use CodeMirror for rendering our code.
 *
 * @exports feature/file-content/side-by-side-diff-view
 */
function(
    $,
    _,
    fn,
    math,
    performance,
    promiseUtil,
    svg,
    Direction,
    DiffHunkMap,
    DiffView,
    DiffFileTypes,
    diffViewSegmentTypes,
    StashLineHandle,
    sbsSynchronizedScroll
) {
    'use strict';

    var ADDED = diffViewSegmentTypes.ADDED;
    var REMOVED = diffViewSegmentTypes.REMOVED;

    /**
     * Manage Side-by-side Diff View and its base functionality.
     *
     * @param {Object} data - diff JSON
     * @param {Object} options - file options
     * @constructor
     */
    function SideBySideDiffView(data, options) {
        options.combineLinkedSegments = true;
        DiffView.apply(this, arguments);
    }
    _.extend(SideBySideDiffView.prototype, DiffView.prototype);

    /**
     * Initialize the Side-by-side Diff
     */
    SideBySideDiffView.prototype.init = function() {
        this.$container.addClass('side-by-side-diff');

        this.$container.append(stash.feature.fileContent.sideBySideDiffView.layout());

        this.fromEditorEl = this.$container.find('.side-by-side-diff-editor-from');
        this.toEditorEl = this.$container.find('.side-by-side-diff-editor-to');

        var commentGutter = this.options.commentContext && this.options.commentContext.getGutterId();
        this._registerGutter({ name: commentGutter,      weight: 100 });
        this._registerGutter({ name: 'line-number-from', weight: 200, fileType: DiffFileTypes.FROM });
        this._registerGutter({ name: 'line-number-to',   weight: 200, fileType: DiffFileTypes.TO });

        this._fromEditor = this._createEditor({
            gutters : this.gutters,
            gutterProvider: this._guttersForFileType.bind(this, DiffFileTypes.FROM)
        }, this.fromEditorEl);

        this._toEditor = this._createEditor({
            gutters : this.gutters,
            gutterProvider: this._guttersForFileType.bind(this, DiffFileTypes.TO)
        }, this.toEditorEl);

        DiffView.prototype.init.call(this);

    };

    /**
     * Prepare the diff view for GC. It's unusable after this.
     */
    SideBySideDiffView.prototype.destroy = function() {
        if (this._detachScrollingBehavior) {
            this._detachScrollingBehavior();
            this._detachScrollingBehavior = null;
        }

        DiffView.prototype.destroy.call(this);

        this._fromEditor = null;
        this._toEditor = null;
    };

    /**
     * Set gutter element for the specified gutter at the specified line.
     *
     * @param {StashLineHandle} lineHandle - line handle as returned from {@link getLineHandle}
     * @param {string} gutterId - ID of the gutter for which to set a marker
     * @param {HTMLElement} el - element to set the gutter to.
     * @returns {StashLineHandle}
     */
    SideBySideDiffView.prototype.setGutterMarker = function (lineHandle, gutterId, el) {
        this._editorForHandle(lineHandle).setGutterMarker(lineHandle._handle, gutterId, this._prepareGutterMarkerElement(el));
        return lineHandle;
    };

    /**
     * Add a CSS class to a specified line
     *
     * @param {StashLineHandle} lineHandle - line handle as returned from {@link getLineHandle}
     * @param {string} whichEl - 'wrap', 'background', or 'text' to specify which element to place the class on
     * @param {string} className - the class to add.
     * @returns {StashLineHandle}
     */
    SideBySideDiffView.prototype.addLineClass = function (lineHandle, whichEl, className) {
        this._editorForHandle(lineHandle).addLineClass(lineHandle._handle, whichEl, className);
        return lineHandle;
    };

    /**
     * Remove a CSS class from a specified line
     *
     * @param {StashLineHandle} lineHandle - as returned from {@link getLineHandle}
     * @param {string} whichEl - 'wrap', 'background', or 'text' to specify which element to remove the class from
     * @param {string} className - the class to remove.
     * @returns {StashLineHandle}
     */
    SideBySideDiffView.prototype.removeLineClass = function (lineHandle, whichEl, className) {
        this._editorForHandle(lineHandle).removeLineClass(lineHandle._handle, whichEl, className);
        return lineHandle;
    };

    /**
     * Return the text on the line with the given line handle.
     *
     * @param {StashLineHandle} lineHandle - as returned from {@link getLineHandle}
     * @returns {string}
     */
    SideBySideDiffView.prototype.getLine = function (lineHandle) {
        return lineHandle._handle.text;
    };

    SideBySideDiffView.prototype.operation = function(func) {
        var self = this;
        // run in an operation for both editors so they both hold their updates til the end.
        return this._fromEditor.operation(function() {
            return self._toEditor.operation(fn.arity(func, 0).bind(null));
        });
    };

    /**
     * Update the editors when something has changed (e.g., size of the editor).
     */
    SideBySideDiffView.prototype.refresh = function() {
        this._fromEditor.refresh();
        this._toEditor.refresh();
        this.trigger('resize');
    };

    /**
     * @see {@link DiffView:_acceptModification}
     * @protected
     */
    SideBySideDiffView.prototype._acceptModification = function(diff, lineInfos, changeType) {
        var self = this;
        if (changeType !== 'INITIAL') {
            throw new Error('Unrecognized change type: ' + changeType);
        }

        var fromLineInfos = _.filter(lineInfos, function (l) {
            return l.line.lineType !== ADDED;
        });
        var toLineInfos = _.filter(lineInfos, function (l) {
            return l.line.lineType !== REMOVED;
        });

        this._fromEditor.setValue(DiffView._combineTexts(fromLineInfos));
        this._toEditor.setValue(DiffView._combineTexts(toLineInfos));

        // must be deferred because there is a file-content spinner that affects page height and our calculations are screwed.
        var scrollPromise = promiseUtil.delay(function(){
            return $.when(self._attachScrollBehavior(lineInfos)).then(function(syncScrollingInfo) {
                if (syncScrollingInfo) { // if !destroyed
                    self.setupHunkmaps(syncScrollingInfo.linkedFromAndToRegions);
                }
            });
        })();

        function isPairedToChange(prevLineInfo, lineInfo) {
            // a context line after a context line from a different segment means there is a change in the other editor
            return prevLineInfo.segment !== lineInfo.segment &&
                   prevLineInfo.line.lineType === diffViewSegmentTypes.CONTEXT &&
                   lineInfo.line.lineType === diffViewSegmentTypes.CONTEXT;
        }

        function setupLines(lineInfos, editor, handleProp) {
            var prevLineInfo;
            _.forEach(lineInfos, function(lineInfo, i) {
                var handle = editor.getLineHandle(i);
                lineInfo._setHandle(handleProp, new StashLineHandle(handleProp, lineInfo.line.lineType, lineInfo.line.lineNumber, handle));

                if (prevLineInfo && isPairedToChange(prevLineInfo, lineInfo)) {
                    editor.addLineClass(handle, 'wrap', 'paired-with-change');
                }
                prevLineInfo = lineInfo;
            });
        }

        setupLines(fromLineInfos, this._fromEditor, DiffFileTypes.FROM);
        setupLines(toLineInfos, this._toEditor, DiffFileTypes.TO);
        return scrollPromise;
    };

    var editorForFileType = {};
    editorForFileType[DiffFileTypes.FROM] = '_fromEditor';
    editorForFileType[DiffFileTypes.TO] = '_toEditor';

    /**
     * @see {@link DiffView:_editorForHandle}
     * @protected
     */
    SideBySideDiffView.prototype._editorForHandle = function (handle) {
        return this[editorForFileType[handle.fileType]];
    };

    /**
     * @see {@link DiffView:_getEditors}
     * @private
     */
    SideBySideDiffView.prototype._getEditors = function() {
        return [this._toEditor, this._fromEditor];
    };

    /**
     * Return the segment that is currently at the 'focus point' in the viewport
     *
     * @param {number} diffViewOffset px offset of the diffview from the top of the viewport. We offset our values by this.
     * @returns {Object} segment
     * @private
     */
    SideBySideDiffView.prototype._getFocusSegment = function(diffViewOffset) {
        var focusOffset = this._fromEditor.getScrollInfo().clientHeight * this.options.focusPoint;
        if (diffViewOffset > focusOffset) {
            return null; // too far up for anything to be focused.
        }

        // HACK: for some reason IE reports one pixel _less_ than other browsers. Foregoing the effort of finding out why
        // and adding 1px to the current focus line. Given that we're expecting the first pixel of each line here, we
        // still have a leeway of 16px out of 17px per line before we start skipping segments.
        var sprinkleOfIEMagic = 1;

        var scrollPos = focusOffset + this._combinedScrollable._getScrollTop() - diffViewOffset + sprinkleOfIEMagic;
        scrollPos = Math.ceil(scrollPos); // CodeMirror ceils its values, so we do that to avoid subpixel errors.
        var focusedRegion = _.find(this._combinedRegions, function(region) {
            return region._getOffset() <= scrollPos &&
                   (region._getOffset() + region.getHeight()) > scrollPos;
        });

        // if the diff is too short, use the last region.
        return (focusedRegion || _.last(this._combinedRegions))._seg;
    };

    /**
     * Scroll to the location that puts the first line of the given segment at the 'focus point'
     *
     * @param {Object[]} segments
     * @private
     */
    SideBySideDiffView.prototype._setFocusSegment = function(segments) {
        var focusedRegions = _.filter(this._combinedRegions, function(region) {
            return _.contains(segments, region._seg);
        });

        var focusOffset = this._fromEditor.getScrollInfo().clientHeight * this.options.focusPoint;
        this._scrollToSourcePosition(null, focusedRegions[0]._getOffset() - focusOffset);

        var lines = _.chain(focusedRegions)
                    .pluck('_linkedRegions').flatten()
                    .pluck('_seg').pluck('lines').flatten()
                    .value();

        var getLineHandle = this.getLineHandle.bind(this);

        function handles(lines, fileType) {
            var excludedLineType = fileType === 'FROM' ? ADDED : REMOVED;
            var fileTypeLines = _.filter(lines, function(line) {
                return line.lineType !== excludedLineType;
            });
            return fileTypeLines.map(function(line) {
                return getLineHandle({
                    lineType : line.lineType,
                    lineNumber : line.lineNumber,
                    fileType : fileType
                });
            });
        }

        var self = this;
        this.operation(function() {
            self._markLinesFocused([
                {editor: self._fromEditor, handles: handles(lines, 'FROM')},
                {editor: self._toEditor, handles: handles(lines, 'TO')}
            ]);
        });
    };

    /**
     * Find the next comment anchor in a given direction, from the current focus
     *
     * @param {Direction} direction
     * @param {Array<Object>} anchors
     * @param {Object} focusedAnchorInfo
     * @private
     */
    SideBySideDiffView.prototype._findNextAnchor = function(direction, anchors, focusedAnchorInfo) {
        var anchorsByFileType = _.groupBy(anchors, fn.dot('_fileType'));
        var nextFromAnchorInfo = anchorsByFileType.FROM &&
            this._findNextAnchorInEditor(this._fromEditor, anchorsByFileType.FROM, direction, focusedAnchorInfo);
        var nextToAnchorInfo = anchorsByFileType.TO &&
            this._findNextAnchorInEditor(this._toEditor, anchorsByFileType.TO, direction, focusedAnchorInfo);

        if (!nextFromAnchorInfo || !nextToAnchorInfo) {
            // if there's only one, it's that one
            return nextFromAnchorInfo || nextToAnchorInfo || null;
        } else {
            // When moving downward, the FROM comes first when equal. When moving upward, TO comes first.
            if ((direction === Direction.UP   && nextToAnchorInfo.offset < nextFromAnchorInfo.offset) ||
                (direction === Direction.DOWN && nextToAnchorInfo.offset >= nextFromAnchorInfo.offset)) {
                return nextFromAnchorInfo;
            } else {
                return nextToAnchorInfo;
            }
        }
    };


    /**
     * Set up the hunk maps for the From and To sides of the editor.
     * @param linkedRegions
     */
    SideBySideDiffView.prototype.setupHunkmaps = function (linkedRegions) {
        var fromRegions = linkedRegions.map(_.first);
        var toRegions = linkedRegions.map(_.last);
        var focusPoint = this.options.focusPoint;
        /**
         * Scroll the given editor to a relative position
         * @param {CodeMirror} editor
         * @param {number} fraction
         */
        function scrollTo(editor, fraction) {
            var scrollInfo = editor.getScrollInfo();
            var contentHeight = scrollInfo.height;
            var viewportHeight = scrollInfo.clientHeight;
            // position our clicked location {options.focusPoint} from the top
            // Also bound it to the top of the editor.
            editor.scrollTo(null, Math.max(0, (contentHeight * fraction) - (viewportHeight * focusPoint)));
        }

        var fromHunkMap = new DiffHunkMap(this.fromEditorEl, fromRegions, { scrollToFn: _.partial(scrollTo, this._fromEditor) });
        var toHunkMap = new DiffHunkMap(this.toEditorEl, toRegions, { scrollToFn: _.partial(scrollTo, this._toEditor) });

        var hunkMaps = [fromHunkMap, toHunkMap];

        var redraw = _.debounce(_.invoke.bind(_, hunkMaps, 'redraw'), 100);

        this._destroyables = this._destroyables.concat(hunkMaps);

        this.on('widgetAdded', redraw);
        this.on('widgetChanged', redraw);
        this.on('widgetCleared', redraw);
        this.on('resize', redraw);

        this._fromEditor.on('scroll', function(editor) {
            fromHunkMap.diffScrolled(editor.getScrollInfo());
        });
        this._toEditor.on('scroll', function(editor) {
            toHunkMap.diffScrolled(editor.getScrollInfo());
        });
    };

    /**
     * Set up a few different behaviors:
     * - Synchronized scrolling between the left and right sides
     * - Page-level scroll forwarding for a 'full-screen' mode.
     * - scrolls the editor to the first real change in the file.
     *
     * @param {LineInfo[]} lineInfos
     * @returns {SyncScrollingInfo}
     * @private
     */
    SideBySideDiffView.prototype._attachScrollBehavior = function (lineInfos) {
        var self = this;
        var fromEditor = this._fromEditor;
        var toEditor = this._toEditor;

        if (!fromEditor) return; // destroyed before we started

        // set up synchronized scrolling between the two sides.
        var syncScrollingInfo = sbsSynchronizedScroll.setupScrolling(this, lineInfos, fromEditor, toEditor, {
            includeCombinedScrollable : true,
            focusHeightFraction: self.options.focusPoint
        });

        // store these for use in segment navigation
        this._combinedScrollable = syncScrollingInfo.combinedScrollable;
        this._combinedRegions = syncScrollingInfo.combinedRegions;

        var internalScrollEvent = $.Callbacks();

        syncScrollingInfo.combinedScrollable.on('scroll', function(x, y, source) {
            if (source === 'sync') { // send synchronization scrolls back up to the window.
                internalScrollEvent.fire(x, y);
            } else {
                // a 'native' scroll comes down from the window, so no point firing an event to send it back up.
            }
        });

        // Link up the combined scrollable from our sync scrolling to the window
        // - Whenever the page is scrolled, it will call our scroll() function and we need to forward that to the combined scrollable.
        // - Whenever the page is resized, it'll let us know so we can resize ourself as needed.
        // - We can also call the onSizeChange and onInternalScroll callbacks it adds whenever we need the page to update based on our
        //   scroll location or size changes.
        var $editorColumns = self.$container.children('.diff-editor, .segment-connector-column');
        this._requestWindowScrolls({
            scrollSizing : function() {
                return syncScrollingInfo.combinedScrollable.getScrollInfo();
            },
            scroll : function(x, y) {
                if (y != null) { // ignore horizontal changes
                    syncScrollingInfo.combinedScrollable.scrollToNative(null, y);
                    // don't fire internalScrollEvent, which would just tell the window to register the same scroll
                    // that the window just told us about.
                }
            },
            resize : function(width, height) {
                // ignore width changes
                syncScrollingInfo.combinedScrollable.setClientHeight(height);
                $editorColumns.height(height);
                self.refresh();
            },
            onSizeChange : function(fn) {
                self.on('resize', fn);
            },
            onInternalScroll : function(fn) {
                internalScrollEvent.add(fn);
            }
        }).then(function() {
            // Refresh the editor once scroll control has been surrendered to ensure all widgets update their size
            self.refresh();
        });

        // draw paths between the left and right side
        this._initSegmentLinking(syncScrollingInfo.linkedFromAndToRegions);

        // scroll to the first change in the file so they're not looking at all white.
        this._scrollEditorToFirstChange();

        this._detachScrollingBehavior = function() {
            syncScrollingInfo.destroy();
        };

        return syncScrollingInfo;
    };

    /**
     * Setup the center SVG segment-linking column to handle resizing and scrolling.
     *
     * @param {CodeMirrorRegion[][]} linkedRegionsList - a list of regions from each scrollable (from, to) that are linked together.
     * @private
     */
    SideBySideDiffView.prototype._initSegmentLinking = function(linkedRegionsList) {
        var $segmentColumn = $('.segment-connector-column');
        var svgEl = svg.createElement('svg', {});
        $segmentColumn.append(svgEl);

        var updateSegmentConnectors = updateSegmentLinkingColumn.bind(null, getLinkableSegments(linkedRegionsList), svgEl, this._getLineClasses);

        var resize = performance.enqueueCapped(requestAnimationFrame, function resize() {
            svgEl.setAttribute('height', $segmentColumn.height());
            svgEl.setAttribute('width', $segmentColumn.width());
            updateSegmentConnectors();
        });

        var updateSegmentConnectorsOnAnimationFrame = performance.enqueueCapped(requestAnimationFrame, updateSegmentConnectors);

        this.on('sync-scroll', updateSegmentConnectors);
        this.on('widgetAdded', updateSegmentConnectorsOnAnimationFrame);
        this.on('widgetChanged', updateSegmentConnectorsOnAnimationFrame);
        this.on('widgetCleared', updateSegmentConnectorsOnAnimationFrame);
        this.on('resize', resize);
        resize();
    };

    /**
     * Return whether a segment contains modification lines.
     *
     * @param {Object} segment
     * @returns {boolean}
     * @private
     */
    function isModification(segment) {
        return segment && segment.type !== diffViewSegmentTypes.CONTEXT;
    }

    /**
     * Return whether a region contains modification lines.
     * @param {CodeMirrorRegion} region
     * @returns {boolean}
     * @private
     */
    function isRegionModification(region) {
        return isModification(region && region._seg) && region._numLines > 0;
    }

    /**
     * Return whether a segment contains a conflict
     *
     * @param {Object} segment
     * @returns {boolean}
     * @private
     */
    function isConflicted(segment) {
        return Boolean(segment.lines[0].conflictMarker);
    }

    /**
     * Return whether a region contains a conflict
     *
     * @param {CodeMirrorRegion|getLinkableSegments.ConflictedRegion} region
     * @returns {boolean}
     * @private
     */
    function isRegionConflicted(region) {
        return region.conflicted || isConflicted(region._seg);
    }

    /**
     * Scroll an editor to the first change, based of a set of segments.
     *
     * @private
     */
    SideBySideDiffView.prototype._scrollEditorToFirstChange = function() {
        var firstChange = this._findNextChange(null, this._allSegments);
        if (firstChange) {
            this._setFocusSegment(firstChange);
        }
    };

    /**
     * Get a list of segments for use in the central SVG column. The list only includes modified segments, and
     * conflicted segments.
     *
     * @param {CodeMirrorRegion[][]} linkedRegionsList - a list of regions from each scrollable (from, to) that are linked together.
     */
    function getLinkableSegments(linkedRegionsList) {

        /**
         * Combined a few adjacent conflict regions into a single one for use in the segment linking column.
         * @param {CodeMirrorRegion} region
         * @returns {getLinkableSegments.ConflictedRegion}
         * @constructor
         * @private
         */
        function ConflictedRegion(region) {
            if (!(this instanceof ConflictedRegion)) {
                return new ConflictedRegion(region);
            }
            this._regions = [region];
            this.conflicted = true;
            this.classesInfo = {
                lineType : region._seg.type,
                conflictMarker : region._seg.lines[0].conflictMarker
            };
            this.getOffsetTop = function() {
                return this._regions[0].getOffsetTop();
            };
            this.getHeight = function() {
                return _.chain(this._regions).invoke('getHeight').reduce(math.add).value();
            };
            this.push = function(region) {
                this._regions.push(region);
            };
        }

        return linkedRegionsList.reduce(function combineConflicts(memo, linkedRegions) {
            var previousLinkedRegions = memo.previous;
            var previousConflict = previousLinkedRegions && _.some(previousLinkedRegions, isRegionConflicted);
            var currentConflict = _.some(linkedRegions, isRegionConflicted);

            if (currentConflict && previousConflict) { // join with the previous conflict region
                _.forEach(previousLinkedRegions, function(conflictedRegion, i) {
                    conflictedRegion.push(linkedRegions[i]);
                });
                return memo;
            }

            if (currentConflict) { // create a new conflict region to sum up all the upcoming conflicts
                linkedRegions = _.map(linkedRegions, ConflictedRegion);
            }

            memo.previous = linkedRegions;
            memo.regions.push(linkedRegions);
            return memo;
        }, {
            previous : null,
            regions: []
        }).regions.filter(function filterContext(linkedRegions) {
            return linkedRegions.some(fn.or(isRegionModification, isRegionConflicted));
        });
    }

    /**
     * @param {CodeMirrorRegion[][]} linkedRegionsList - a list of regions from each scrollable (from, to) that are linked together.
     * @param {Element} svgEl - <svg> element to populate
     * @param {Function} getLineClasses - a function that returns a string with the appropriate CSS classes, given some metadata about a line.
     */
    function updateSegmentLinkingColumn(linkedRegionsList, svgEl, getLineClasses) {
        var svgStyle;
        var height = svgEl.offsetHeight || parseFloat((svgStyle = window.getComputedStyle(svgEl)).height);
        var width = svgEl.offsetWidth || parseFloat(svgStyle.width);
        var pastWidth = width + 1;
        var curvePointLeft = width * 0.4;
        var curvePointRight = width * 0.6;

        var visibleRegionInfo = linkedRegionsList.map(function getInfo(linkedRegions) {
            return linkedRegions.map(function(r) {
                var top = r.getOffsetTop();
                var bottom = top + r.getHeight();

                return {
                    region : r,
                    top : top + 0.5, // SVG points are centered on the middle of the pixel, so the lines are antialiased and blurry. shifting them down by 0.5 pixels realigns them back with the pixel grid and makes them sharp again
                    bottom : bottom + 0.5,
                    above : top < 0,
                    inside : bottom > 0 && top < height,
                    below : bottom > height
                };
            });
        })
        .filter(function isVisible(linkedRegionInfos) {
            return linkedRegionInfos.some(fn.dot('inside')) ||
                (linkedRegionInfos.some(fn.dot('above')) && linkedRegionInfos.some(fn.dot('below')));
        });

        function getPath(fromRegionInfo, toRegionInfo) {
            return new svg.PathBuilder()
                   .moveTo(-1, fromRegionInfo.top)
                   .curve(curvePointLeft, fromRegionInfo.top,
                          curvePointRight, toRegionInfo.top,
                          pastWidth, toRegionInfo.top)
                   .lineTo(pastWidth, toRegionInfo.bottom)
                   .curve(curvePointRight, toRegionInfo.bottom,
                          curvePointLeft, fromRegionInfo.bottom,
                          -1, fromRegionInfo.bottom)
                   .close()
                   .build();
        }

        function getClassesInfo(regionInfo, otherRegionInfo) {
            if (regionInfo.region.classesInfo) {
                return regionInfo.region.classesInfo;
            }
            var firstLineInfo = regionInfo.region._lineInfos[0];
            var otherFirstLineInfo = otherRegionInfo.region._lineInfos[0];
            return firstLineInfo && firstLineInfo.line || {
                conflictMarker : null,
                lineType : otherFirstLineInfo.line.lineType
            };
        }
        function getClasses(fromRegionInfo, toRegionInfo) {
            var fromInfo = getClassesInfo(fromRegionInfo, toRegionInfo);
            var toInfo = getClassesInfo(toRegionInfo, fromRegionInfo);
            var allClasses = getLineClasses(fromInfo.lineType, fromInfo.conflictMarker, false) + ' ' +
                             getLineClasses(toInfo.lineType, toInfo.conflictMarker, false);
            return _.chain(allClasses.split(/\s+/)).unique().without('line').value().join(' ');
        }

        var templateData = visibleRegionInfo.map(fn.spread(function(fromRegionInfo, toRegionInfo) {
            return {
                path : getPath(fromRegionInfo, toRegionInfo),
                extraClasses : getClasses(fromRegionInfo, toRegionInfo)
            };
        }));

        while(svgEl.hasChildNodes()) {
            svgEl.removeChild(svgEl.firstChild);
        }

        var isAddedAndRemoved = function(classes) {
            return (classes.indexOf('added') !== -1) && (classes.indexOf('removed') !== -1);
        };

        var getSvgGradient = _.once(function(gradientId){
            //Would be nice to move the offset to CSS with the `stop-color`, but it didn't like that
            var stops = [{
                'class': 'removed',
                offset: '0%'
            },
            {
                'class': 'removed',
                offset: '30%'
            },
            {
                'class': 'added',
                offset: '70%'
            },
            {
                'class': 'added',
                offset: '100%'
            }];

            stops = _.map(stops, svg.createElement.bind(svg, 'stop'));

            return _.reduce(stops, function(grad, stop) {
                grad.appendChild(stop);
                return grad;
            }, svg.createElement('linearGradient', {
                'id': gradientId
            }));
        });

        var gradientId = 'added-and-removed-svg-gradient';
        var fragment = templateData.map(function(data) {
            var props = {
                'class' : 'segment-connector ' + data.extraClasses,
                d : data.path
            };

            if (isAddedAndRemoved(data.extraClasses)) {
                //This sucks, but Firefox won't let you set a svg gradient fill via CSS.
                _.extend(props, {
                    fill: 'url(#'+ gradientId + ')'
                });
            }

            return svg.createElement('path', props);
        }).concat(getSvgGradient(gradientId)) //Add the gradient definition as the last element
        .reduce(function(frag, pathEl) {
            frag.appendChild(pathEl);
            return frag;
        }, document.createDocumentFragment());

        svgEl.appendChild(fragment);
    }

    /**
     * Get unique gutters by name and side
     *
     * @returns {Array<object>}
     */
    SideBySideDiffView.prototype.getGutters = _.partial(DiffView.prototype.getGutters, ['name', 'fileType']);

    /**
     * Register a gutter to be added to the editors.
     * @param {string} name
     * @param {object} options
     * @param {DiffFileType} [options.fileType]
     */
    SideBySideDiffView.prototype.registerGutter = function(name, options) {
        var self = this;
        if (!this._fromEditor || !this._toEditor) {
            return; //destroyed
        }
        var editors;
        // If a side has been specified, only add this gutter to that side.
        if (options.fileType) {
            editors = [
                [ options.fileType, this[editorForFileType[options.fileType]] ]
            ];
        }
        // for an unspecified side, add the gutter to both editors.
        else {
            editors = [
                [DiffFileTypes.FROM, this._fromEditor],
                [DiffFileTypes.TO, this._toEditor]
            ];
        }

        this._registerGutter({name: name, weight: options.weight, fileType: options.fileType });

        editors.forEach(fn.spread(function(fileType, editor) {
            editor.setOption('gutters', _.pluck(self._guttersForFileType(fileType), 'name'));
        }));

    };

    /**
     * Find all gutters for a given side.
     *
     * This will also return all gutters that do not specify a side.
     *
     * @param {DiffFileTypes} fileType
     * @returns {Array<Object>}
     * @private
     */
    SideBySideDiffView.prototype._guttersForFileType = function(fileType){
        var filtered = this.getGutters().filter(function(gutter){
            return gutter.fileType === fileType || typeof gutter.fileType === 'undefined';
        });
        return filtered;
    };



    return SideBySideDiffView;
});
