define('feature/file-content/request-diff', [
    'jquery',
    'stash/api/util/navbuilder',
    'util/ajax',
    'feature/file-content/diff-view-options'
], function (
    $,
    navbuilder,
    ajax,
    diffViewOptions
) {

    // See notes on requestDiff. We cache to avoid multiple server requests for the same information by different handlers.
    var cache = {};
    var DEFAULT_CONTEXT_LINES = -1; // a negative value will set the context lines to the system default

    // Data is returned as an array due to the REST resource supporting multiple diffs
    // The first will not necessarily be the one we want. We want the first with the destination
    // set to the selected file. This avoids choosing the wrong diff where a file has been modified
    // and copied in the same changeset, for example.
    function getMatchingDiff(fileChangeJSON, data) {
        if ($.isArray(data.diffs) && data.diffs.length) {
            data = _.find(data.diffs, function(diff) {
                if (diff.destination) {
                    return diff.destination.toString === fileChangeJSON.path.toString();
                } else if (fileChangeJSON.srcPath) {
                    return diff.source.toString === fileChangeJSON.srcPath.toString();
                }
                return false;
            }) || data.diffs[0]; //Or the first diff if none were found (this shouldn't happen)
        }
        return data;
    }

    /**
     * A builder to generate the diff URL.
     *
     * @param {JSON.FileChangeJSON} fileChangeJSON a fileChange object describing the change
     * @returns {stash/api/util/navbuilder.Builder} a builder generating the diff URL
     * @private
     */
    function requestDiffUrlBuilder(fileChangeJSON) {
        var baseUrlBuilder = navbuilder.rest()
            .project(fileChangeJSON.repository.project.key)
            .repo(fileChangeJSON.repository.slug);

        var pullRequest = fileChangeJSON.commitRange.pullRequest;
        var diffUrlBuilder = pullRequest ?
            baseUrlBuilder.pullRequest(pullRequest.id) :
            baseUrlBuilder.changeset(fileChangeJSON.commitRange);
        return diffUrlBuilder.diff(fileChangeJSON);
    }

    /**
     * Request diff information from the server. Requests are cached for the remainder of an event loop after they are resolved.
     * This helps with performance of multiple handlers requesting the same data.
     *
     * @param {JSON.FileChangeJSON} fileChange a fileChange object describing the change
     * @param {Object} options additional options
     * @returns {Promise} a promise that resolves to the diff JSON returned form the server.
     */
    function requestDiff(fileChange, options) {
        var fileChangeJSON = fileChange.toJSON ? fileChange.toJSON() : fileChange;
        if (fileChangeJSON.diff) {
            return $.Deferred().resolve($.extend({
                lineComments: options.lineComments || [],
                fileComments: options.fileComments || []
            }, fileChangeJSON.diff));
        }

        options = options || {};
        var ignoreWhitespace = options.hasOwnProperty('ignoreWhitespace') ?
                                   options.ignoreWhitespace :
                                   diffViewOptions.get('ignoreWhitespace');

        var contextLines = isNaN(Number(options.contextLines)) ? DEFAULT_CONTEXT_LINES : Math.floor(options.contextLines);

        var urlBuilder = (options.diffUrlBuilder || requestDiffUrlBuilder)(fileChangeJSON);
        var url = urlBuilder
            .withParams({
                avatarSize: stash.widget.avatarSizeInPx({ size: options.avatarSize || 'medium' }),
                markup: true,
                whitespace: ignoreWhitespace ? 'ignore-all' : '',
                contextLines: contextLines,
                withComments: options.withComments
            })
            .build();

        if (cache.hasOwnProperty(url) && cache[url].state() !== 'rejected') {
            return cache[url];
        }

        var xhr = ajax.rest({
            url: url,
            statusCode : options.statusCode || ajax.ignore404WithinRepository()
        });

        var piped = xhr.then(function(data) {
            if (data.errors && data.errors.length) {
                return $.Deferred().rejectWith(this, [this, null, null, data]);
            } else {
                data = getMatchingDiff(fileChangeJSON, data);
            }
            setTimeout(function() {
                delete cache[url];
            });
            return data;
        });

        cache[url] = piped.promise(xhr);
        return cache[url];
    }

    return requestDiff;
});
