package com.atlassian.webdriver.utils;

import com.atlassian.webdriver.LifecycleAwareWebDriverGrid;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.function.TriConsumer;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WrapsDriver;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.openqa.selenium.ie.InternetExplorerDriver;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * @since 2.1
 */
public final class WebDriverUtil
{
    private static final Logger log = LoggerFactory.getLogger(WebDriverUtil.class);

    private WebDriverUtil()
    {
        throw new AssertionError(WebDriverUtil.class.getName() + " is not supposed to be instantiated");
    }

    public static boolean isFirefox(@Nonnull WebDriver driver)
    {
        return getUnderlyingDriver(driver) instanceof FirefoxDriver;
    }

    public static boolean isChrome(@Nonnull WebDriver driver)
    {
        return getUnderlyingDriver(driver) instanceof ChromeDriver;
    }

    public static boolean isIE(@Nonnull WebDriver driver)
    {
        return getUnderlyingDriver(driver) instanceof InternetExplorerDriver;
    }

    /**
     * Retrieve the underlying {@link WebDriver} instance.
     *
     * @param driver the driver to examine
     * @return the underlying web driver instance that is actually driving the test. This can be the same instance
     * as {@literal driver}, or some other instance that is wrapped by {@literal driver}.
     */
    @Nonnull
    public static WebDriver getUnderlyingDriver(@Nonnull WebDriver driver)
    {
        requireNonNull(driver, "driver");
        while (driver instanceof WrapsDriver)
        {
            driver = ((WrapsDriver) driver).getWrappedDriver();
        }
        return driver;
    }

    /**
     * Check whether the underlying driver is instance of {@literal type}. This method attempts to retrieve the underlying
     * web driver before performing the instance check.
     *
     * @param driver driver to check
     * @param type type to check
     * @return {@literal true}, if the underlying driver is an instance of {@literal type}, in which case
     * {@link #as(WebDriver, Class)} may be safely used on it
     * @see #getUnderlyingDriver(WebDriver)
     */
    public static boolean isInstance(@Nonnull WebDriver driver, @Nonnull Class<?> type)
    {
        return type.isInstance(getUnderlyingDriver(driver));
    }

    /**
     * Get the underlying web driver instance from {@literal driver} as instance of {@literal type}.
     *
     * @param driver driver to examine
     * @param type type to cast to
     * @param <T> type parameter
     * @return the underlying driver instance as T
     * @throws ClassCastException if {@link #isInstance(WebDriver, Class)} for given arguments returns {@literal false}
     * @see #isInstance(org.openqa.selenium.WebDriver, Class)
     */
    public static <T> T as(@Nonnull WebDriver driver, @Nonnull Class<T> type)
    {
        return type.cast(getUnderlyingDriver(driver));
    }

    /**
     * Parses capabilities from input string
     * @since 2.3
     */
    @Nonnull
    public static DesiredCapabilities createCapabilitiesFromString(@Nullable String capabilitiesList) {
        DesiredCapabilities capabilities = new DesiredCapabilities();
        setPropertiesFromString(capabilitiesList, capabilities,
                MutableCapabilities::setCapability);
        return capabilities;
    }

    /**
     * Parses properties from input string and sets them on the target object
     * @since 3.5
     */
    public static <T> void setPropertiesFromString(String propertiesString, T target, TriConsumer<T, String, String> setter) {
        if (!StringUtils.isEmpty(propertiesString)) {
            for (String property : propertiesString.split(";")) {
                String[] nameVal = property.split("=", 2);
                if (nameVal.length == 2) {
                    String name = nameVal[0].trim();
                    String valStr = nameVal[1].trim();
                    setter.accept(target, name, valStr);
                } else {
                    log.error("Invalid property string: {}. Expected format: key1=value1;key2=value2", property);
                }
            }
        }
    }

}
