package com.atlassian.webdriver;

import com.browserstack.local.Local;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import static com.atlassian.webdriver.WebDriverProperties.WEBDRIVER_CAPABILITIES;
import static org.apache.commons.lang3.StringUtils.isEmpty;

public class BrowserstackWebDriverFactory {
    private static final Logger log = LoggerFactory.getLogger(BrowserstackWebDriverFactory.class);

    private static final String[] BS_USER_KEYS = {"BROWSERSTACK_USER", "BROWSERSTACK_USERNAME"};
    private static final String[] BS_AUTH_KEYS = {"BROWSERSTACK_ACCESS_KEY", "BROWSERSTACK_ACCESSKEY", "BROWSERSTACK_KEY", "BROWSERSTACK_AUTH"};
    private static final String[] BUILD_KEYS = {"CI_BUILD_NUMBER", "BAMBOO_BUILD_NUMBER", "BITBUCKET_BUILD_NUMBER"};

    private static final String NAME = "browserstack";

    private final BrowserstackWebDriverBuilder builder;

    public BrowserstackWebDriverFactory() {
        final Optional<String> username = findDefinedEnv(BS_USER_KEYS);
        final Optional<String> authkey = findDefinedEnv(BS_AUTH_KEYS);

        if (!username.isPresent()) {
            throw new IllegalStateException("Browserstack username must be set " +
                    "in one of the following environment variables: " + Arrays.toString(BS_USER_KEYS));
        }

        if (!authkey.isPresent()) {
            throw new IllegalStateException("Browserstack authentication key must be set " +
                    "in one of the following environment variables: " + Arrays.toString(BS_AUTH_KEYS));
        }

        builder = new BrowserstackWebDriverBuilder(username.get(), authkey.get())
                .withBrowser(System.getProperty("browserstack.browser"))
                .withBrowserVersion(System.getProperty("browserstack.browser.version"))
                .withOS(System.getProperty("browserstack.os"))
                .withOSVersion(System.getProperty("browserstack.os.version"))
                .withProjectName(System.getProperty("browserstack.project"))
                .withTestName(System.getProperty("browserstack.name"))
                .withIdleTimeout(System.getProperty("browserstack.idleTimeout"))
                .withAcceptSslCerts(System.getProperty("acceptSslCerts"))
                .withBuildName(findDefinedEnv(BUILD_KEYS).orElseGet(() -> new Date().toString()));

        // pass additional options if such system properties are set
        setFromSystemProperty(builder::withScreenResolution, "browserstack.resolution");
        setFromSystemProperty(builder::withConsoleLogs, "browserstack.console");
        setFromSystemProperty(builder::withVisualLogs, "browserstack.debug");
        setFromSystemProperty(builder::withNetworkLogs, "browserstack.networkLogs");
        setFromSystemProperty(builder::withVideoLogs, "browserstack.video");

        // pass even more options if defined in the "-Dwebdriver.capabilities" property
        final String capabilitiesString = WEBDRIVER_CAPABILITIES.getSystemProperty();
        log.info("Loading custom capabilities: " + capabilitiesString);
        setFromMap(builder::withCapability, splitPropertyString(capabilitiesString));
    }

    public WebDriverContext getDriverContext(Local local) throws Exception {
        return builder
                .withLocalConnection(local)
                .withLocalIdentifier("atl-selenium_" + new Date().toInstant().toString())
                .build();
    }

    public static boolean matches(String browserProperty) {
        return NAME.equalsIgnoreCase(browserProperty);
    }

    private static Optional<String> findDefinedEnv(String... keys) {
        for (String key : keys) {
            final String val = System.getenv(key);
            if (StringUtils.isNotBlank(val)) {
                return Optional.of(val);
            }
        }
        log.debug("None of the environment variables has been found: " + Arrays.toString(keys));
        return Optional.empty();
    }

    /**
     * If a system property of given name is defined, pass its value to the callback function.
     */
    private void setFromSystemProperty(Consumer<String> callback, String propertyName) {
        final String value = System.getProperty(propertyName);
        if (value != null) {
            log.debug("Passing system property: " + propertyName);
            callback.accept(value);
        }
    }

    /**
     * Iterate over all map and pass their key-value pairs to the callback function.
     */
    private void setFromMap(BiConsumer<String, String> callback, Map<String, String> properties) {
        properties.forEach(callback);
    }

    /**
     * Split "key1=value1;key2=value2;..." into a map.
     */
    private Map<String, String> splitPropertyString(String capabilitiesString) {
        final Map<String, String> properties = new HashMap<>();
        if (!isEmpty(capabilitiesString)) {
            // handle keys with empty values
            Arrays.stream(capabilitiesString.split(";"))
                    .map(cap -> cap.split("="))
                    .filter(nameVal -> nameVal.length > 0)
                    .forEach(nameVal ->
                            properties.put(nameVal[0], nameVal.length > 1 ? nameVal[1] : null));
        }
        return properties;
    }
}
