package com.atlassian.sal.core.pluginsettings;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Stream;

import com.atlassian.sal.core.pluginsettings.AbstractStringPluginSettings.CorruptPluginSettingValueException;
import com.atlassian.sal.core.pluginsettings.AbstractStringPluginSettings.UnsupportedPluginSettingValueTypeException;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static com.atlassian.sal.core.pluginsettings.EscapeUtils.NEW_LINE;
import static com.atlassian.sal.core.pluginsettings.EscapeUtils.VERTICAL_TAB;
import static com.atlassian.sal.core.pluginsettings.EscapeUtils.escape;
import static com.atlassian.sal.core.pluginsettings.EscapeUtils.unescape;
import static com.atlassian.sal.core.pluginsettings.PluginSettingStringSerializer.StringSerializationMode.LEGACY;
import static com.atlassian.sal.core.pluginsettings.PluginSettingStringSerializer.StringSerializationMode.SAFE;

/**
 * @since 8.0
 */
public class PluginSettingStringSerializer {

    private static final String STR_IDENTIFIER = "#java.lang.String";
    private static final String BOOL_IDENTIFIER = "#java.lang.Boolean";
    private static final String LONG_IDENTIFIER = "#java.lang.Long";
    private static final String INT_IDENTIFIER = "#java.lang.Integer";
    private static final String LIST_IDENTIFIER = "#java.util.List";
    private static final String SET_IDENTIFIER = "#java.util.Set";
    private static final String MAP_IDENTIFIER = "#java.util.Map";
    private static final String PROPERTIES_IDENTIFIER = "#java.util.Properties";
    private static final String PROPERTIES_ENCODING = "ISO8859_1";

    static Object deserialize(String val) {
        return deserialize(val, SAFE);
    }

    public static Object deserialize(String val, StringSerializationMode stringSerializationMode) {
        if (val == null) {
            return null;
        } else if (stringSerializationMode == SAFE && val.startsWith(STR_IDENTIFIER)) {
            return val.substring(STR_IDENTIFIER.length() + 1);
        } else if (val.startsWith(BOOL_IDENTIFIER)) {
            return Boolean.parseBoolean(val.substring(BOOL_IDENTIFIER.length() + 1));
        } else if (val.startsWith(PROPERTIES_IDENTIFIER)) {
            return deserializeProperties(val);
        } else if (val.startsWith(LIST_IDENTIFIER) || val.startsWith(SET_IDENTIFIER)) {
            return deserializeCollection(val);
        } else if (val.startsWith(MAP_IDENTIFIER)) {
            return deserializeMap(val);
        }
        try {
            if (val.startsWith(INT_IDENTIFIER)) {
                return Integer.parseInt(val.substring(INT_IDENTIFIER.length() + 1));
            } else if (val.startsWith(LONG_IDENTIFIER)) {
                return Long.parseLong(val.substring(LONG_IDENTIFIER.length() + 1));
            }
        } catch (NumberFormatException e) {
            throw new CorruptPluginSettingValueException("Unparsable number type value", e);
        }
        if (stringSerializationMode == SAFE) {
            throw new CorruptPluginSettingValueException("Unrecognized value type: " + val);
        }
        return val;
    }

    static Properties deserializeProperties(String val) {
        var props = new Properties();
        try {
            props.load(new ByteArrayInputStream(val.getBytes(PROPERTIES_ENCODING)));
        } catch (IOException e) {
            throw new CorruptPluginSettingValueException("Unable to deserialize properties", e);
        }
        return props;
    }

    static Collection<String> deserializeCollection(String val) {
        String[] lines = val.split(String.valueOf(NEW_LINE));
        List<String> rawItems = Arrays.asList(lines).subList(1, lines.length);
        Stream<String> stream = rawItems.stream().map(EscapeUtils::unescape);
        if (val.startsWith(LIST_IDENTIFIER)) {
            return stream.collect(toList());
        } else if (val.startsWith(SET_IDENTIFIER)) {
            return stream.collect(toSet());
        }
        throw new CorruptPluginSettingValueException("Unparsable collection");
    }

    static Map<String, String> deserializeMap(String val) {
        var map = new HashMap<String, String>();
        String[] lines = val.split(String.valueOf(NEW_LINE));
        List<String> rawItems = Arrays.asList(lines).subList(1, lines.length);
        for (String item : rawItems) {
            int tabIndex = item.indexOf(VERTICAL_TAB);
            if (tabIndex == -1) {
                throw new CorruptPluginSettingValueException("Unparsable map item");
            }
            String keyPart = item.substring(0, tabIndex);
            String valuePart = item.substring(tabIndex + 1);
            map.put(unescape(keyPart), unescape(valuePart));
        }
        return map;
    }

    static String serialize(Object val) {
        return serialize(val, SAFE);
    }

    public static String serialize(Object value, StringSerializationMode stringSerializationMode) {
        if (value instanceof String str) {
            return stringSerializationMode == LEGACY ? str : prefixIdentifier(str, STR_IDENTIFIER);
        } else if (value instanceof Boolean bool) {
            return prefixIdentifier(bool, BOOL_IDENTIFIER);
        } else if (value instanceof Long longValue) {
            return prefixIdentifier(longValue, LONG_IDENTIFIER);
        } else if (value instanceof Integer integer) {
            return prefixIdentifier(integer, INT_IDENTIFIER);
        } else if (value instanceof Collection<?> collection) {
            return serializeCollection(collection);
        } else if (value instanceof Properties properties) {
            return serializeProperties(properties);
        } else if (value instanceof Map<?, ?> map) {
            return serializeMap(map);
        }
        throw new UnsupportedPluginSettingValueTypeException("Property type: " + value.getClass() + " not supported");
    }

    static String serializeCollection(Collection<?> collection) {
        if (collection instanceof List<?> list) {
            return serializeList(list);
        } else if (collection instanceof Set<?> set) {
            return serializeSet(set);
        } else {
            throw new UnsupportedPluginSettingValueTypeException(
                    "Property type: " + collection.getClass() + " not supported");
        }
    }

    static String serializeList(List<?> list) {
        String unprefixed = serializeCollectionHelper(list);
        return prefixIdentifier(unprefixed, LIST_IDENTIFIER);
    }

    static String serializeSet(Set<?> set) {
        String unprefixed = serializeCollectionHelper(set);
        return prefixIdentifier(unprefixed, SET_IDENTIFIER);
    }

    private static String serializeCollectionHelper(Collection<?> collection) {
        var sb = new StringBuilder();
        for (Object i : collection) {
            sb.append(escape(i.toString()));
            sb.append(NEW_LINE);
        }
        if (!collection.isEmpty()) sb.setLength(sb.length() - 1);
        return sb.toString();
    }

    static String serializeProperties(Properties properties) {
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        try {
            properties.store(bout, PROPERTIES_IDENTIFIER.substring(1));
            return bout.toString(PROPERTIES_ENCODING);
        } catch (IOException e) {
            throw new UnsupportedPluginSettingValueTypeException("Unable to serialize properties", e);
        }
    }

    static String serializeMap(Map<?, ?> map) {
        var sb = new StringBuilder();
        for (Map.Entry<?, ?> entry : map.entrySet()) {
            sb.append(escape(entry.getKey().toString()));
            sb.append(VERTICAL_TAB);
            sb.append(escape(entry.getValue().toString()));
            sb.append(NEW_LINE);
        }
        if (!map.isEmpty()) sb.setLength(sb.length() - 1);
        return prefixIdentifier(sb.toString(), MAP_IDENTIFIER);
    }

    private static String prefixIdentifier(Object value, String typeIdentifier) {
        return typeIdentifier + NEW_LINE + value;
    }

    /**
     * Data store requires migration when switching modes.
     */
    public enum StringSerializationMode {
        /**
         * Equivalent to serialization technique used in historical {@code AbstractStringPluginSettings}.
         */
        LEGACY,
        /**
         * Prevents unintended coercion of Strings to other types.
         */
        SAFE
    }
}
