package com.atlassian.sal.core.pluginsettings;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;

/**
 * Custom escape utilities to accommodate serialization logic in {@link PluginSettingStringSerializer}.
 */
class EscapeUtils {
    // Not for instantiation.
    private EscapeUtils() {}

    protected static final char VERTICAL_TAB = '\f';
    protected static final char NEW_LINE = '\n';
    protected static final char BACKSLASH = '\\';

    /**
     * This only escapes VERTICAL_TAB, NEW_LINE, and BACKSLASH.
     */
    protected static String escape(String str) {
        if (str == null) {
            return null;
        }
        var writer = new StringWriter(str.length() * 2);
        try {
            escape(writer, str);
        } catch (IOException e) {
            throw new IllegalStateException(
                    "exception while writing to StringWriter (should be impossible in this context)", e);
        }
        return writer.toString();
    }

    private static void escape(Writer out, String str) throws IOException {
        int len = str.length();
        for (int i = 0; i < len; i++) {
            char ch = str.charAt(i);
            switch (ch) {
                case VERTICAL_TAB -> {
                    out.write(BACKSLASH);
                    out.write('f');
                }
                case NEW_LINE -> {
                    out.write(BACKSLASH);
                    out.write('n');
                }
                case BACKSLASH -> {
                    out.write(BACKSLASH);
                    out.write(BACKSLASH);
                }
                default -> out.write(ch);
            }
        }
    }

    public static String unescape(String str) {
        if (str == null) {
            return null;
        }
        try {
            StringWriter writer = new StringWriter(str.length());
            unescape(writer, str);
            return writer.toString();
        } catch (IOException e) {
            throw new IllegalStateException(
                    "exception while writing to StringWriter (should be impossible in this context)", e);
        }
    }

    private static void unescape(Writer out, String str) throws IOException {
        int len = str.length();
        for (int i = 0; i < len; i++) {
            char ch = str.charAt(i);
            if (ch == BACKSLASH) {
                if (i + 1 >= len) {
                    throw new IllegalArgumentException("Invalid escape sequence: trailing backslash at position " + i);
                }
                char nextCh = str.charAt(i + 1);
                switch (nextCh) {
                    case 'f' -> {
                        out.write(VERTICAL_TAB);
                        i++;
                    }
                    case 'n' -> {
                        out.write(NEW_LINE);
                        i++;
                    }
                    case BACKSLASH -> {
                        out.write(BACKSLASH);
                        i++;
                    }
                    default -> throw new IllegalArgumentException(
                            "Invalid escape sequence: \\" + nextCh + " at position " + i);
                }
            } else {
                out.write(ch);
            }
        }
    }
}
