package com.atlassian.sal.core.pluginsettings;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.core.pluginsettings.PluginSettingStringSerializer.StringSerializationMode;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import static com.atlassian.sal.core.pluginsettings.PluginSettingStringSerializer.deserialize;
import static com.atlassian.sal.core.pluginsettings.PluginSettingStringSerializer.serialize;

/**
 * Reference implementation of {@link PluginSettings} backed by a generic String-based {@link KeyValueStore}.
 * All supported value types are serialized to String using {@link PluginSettingStringSerializer}. Note that this
 * class does not enforce any maximum serialized value length unless explicitly configured via the constructor. It is
 * recommended to enforce the limit specified by {@link PluginSettings#MAX_SERIALIZED_VALUE_LENGTH}.
 */
public class AbstractStringPluginSettings implements PluginSettings {

    private static final Logger log = LoggerFactory.getLogger(AbstractStringPluginSettings.class);

    private final KeyValueStore store;
    private final int maxSerializedValueLength;
    private final StringSerializationMode stringSerializationMode;

    /**
     * Default constructor intended for use when extending this class and overriding {@link #getActual(String)},
     * {@link #putActual(String, String)}, and {@link #removeActual(String)}.
     * <p>
     * This constructor was designed for inheritance-based extensibility but is now deprecated in favour of
     * composition-based design patterns for better testability, flexibility, and separation of concerns.
     *
     * @deprecated since 8.0. Prefer composition to inheritance. Instead of extending this class and overriding
     *             the above methods, implement {@link KeyValueStore} and use
     *             {@link #AbstractStringPluginSettings(KeyValueStore, int, StringSerializationMode)}.
     */
    @Deprecated
    public AbstractStringPluginSettings() {
        this.store = new KeyValueStore() {
            @Override
            public void put(String key, String value) {
                AbstractStringPluginSettings.this.putActual(key, value);
            }

            @Override
            public String get(String key) {
                return AbstractStringPluginSettings.this.getActual(key);
            }

            @Override
            public void remove(String key) {
                AbstractStringPluginSettings.this.removeActual(key);
            }
        };
        this.stringSerializationMode = StringSerializationMode.LEGACY;
        this.maxSerializedValueLength = -1;
    }

    /**
     * Creates a new instance backed by the specified {@link KeyValueStore} with configurable maximum serialized value
     * length and string serialization mode.
     *
     * @param store the backing key-value store for persisting settings (must not be {@code null})
     * @param maxSerializedValueLength the maximum allowed length of serialized values, or a non-positive value
     *        to indicate no limit; it's recommended to use {@link PluginSettings#MAX_SERIALIZED_VALUE_LENGTH}
     * @param stringSerializationMode controls how String values are serialized and deserialized.
     *        <ul>
     *        <li>{@link StringSerializationMode#LEGACY LEGACY} - Strings are stored without a type prefix, which
     *        can lead to unintended type coercion on deserialization.</li>
     *        <li>{@link StringSerializationMode#SAFE SAFE} - Strings are stored with a type prefix, ensuring
     *        they are always deserialized as Strings regardless of content.</li>
     *        </ul>
     *        <p><strong>Warning:</strong> If data was previously stored using {@code LEGACY} mode, switching to
     *        {@code SAFE} mode requires migration of existing stored data. Without migration, previously stored
     *        String values (which lack the type prefix) will fail to deserialize and throw a
     *        {@link CorruptPluginSettingValueException}.</p>
     * @since 8.0
     */
    public AbstractStringPluginSettings(
            KeyValueStore store, int maxSerializedValueLength, StringSerializationMode stringSerializationMode) {
        this.store = requireNonNull(store);
        this.maxSerializedValueLength = maxSerializedValueLength;
        this.stringSerializationMode = requireNonNull(stringSerializationMode);
    }

    @Override
    public Object get(String key) {
        validateKey(key);
        return deserialize(store.get(key), stringSerializationMode);
    }

    @Override
    public Object put(String key, Object value) {
        validateKey(key);
        validateValue(value);
        if (value == null || EMPTY.equals(value)) {
            return remove(key);
        }
        String serialized = serialize(value, stringSerializationMode);
        if (maxSerializedValueLength > 0) {
            Validate.inclusiveBetween(
                    1,
                    maxSerializedValueLength,
                    serialized.length(),
                    "Serialized value cannot be longer than %,d characters".formatted(maxSerializedValueLength));
        }
        Object got = null;
        try {
            got = get(key);
        } catch (CorruptPluginSettingValueException e) {
            log.warn("Existing value was corrupt, force-overwriting", e);
        }
        store.put(key, serialized);
        return got;
    }

    @Override
    public Object remove(String key) {
        validateKey(key);
        Object oldValue = null;
        try {
            oldValue = get(key);
            if (oldValue != null) {
                store.remove(key);
            }
        } catch (CorruptPluginSettingValueException e) {
            log.warn("Existing value was corrupt, force-removing", e);
            store.remove(key);
        }
        return oldValue;
    }

    public static void validateKey(String key) {
        Validate.isTrue(key != null, "Key cannot be null");
        Validate.inclusiveBetween(
                1, MAX_KEY_LENGTH, key.length(), "Key cannot be longer than %s characters".formatted(MAX_KEY_LENGTH));
    }

    public static void validateValue(Object val) {
        if (!(val == null
                || val instanceof List
                || val instanceof Set
                || val instanceof Map
                || val instanceof String
                || val instanceof Integer
                || val instanceof Long
                || val instanceof Boolean)) {
            throw new UnsupportedPluginSettingValueTypeException(
                    "Property type: '%s' not supported".formatted(val.getClass().getName()));
        }
        if (val instanceof Collection<?> coll) {
            for (Object o : coll) {
                if (!(o instanceof String)) {
                    throw new UnsupportedPluginSettingValueTypeException(
                            "Collection containing non-String or null object not supported: %s".formatted(o));
                }
            }
        }
        if (val instanceof Map<?, ?> map) {
            for (Map.Entry<?, ?> entry : map.entrySet()) {
                if (!(entry.getKey() instanceof String)) {
                    throw new UnsupportedPluginSettingValueTypeException(
                            "Map containing non-String or null key not supported: %s".formatted(entry.getKey()));
                }
                if (!(entry.getValue() instanceof String)) {
                    throw new UnsupportedPluginSettingValueTypeException(
                            "Map containing non-String or null value not supported: %s".formatted(entry.getValue()));
                }
            }
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AbstractStringPluginSettings that = (AbstractStringPluginSettings) o;
        return Objects.equals(store, that.store)
                && Objects.equals(stringSerializationMode, that.stringSerializationMode);
    }

    @Override
    public int hashCode() {
        return Objects.hash(store, stringSerializationMode);
    }

    /**
     * @since 8.0
     */
    public interface KeyValueStore {
        void put(String key, String value);

        String get(String key);

        void remove(String key);
    }

    /**
     * @since 8.0
     */
    public static class UnsupportedPluginSettingValueTypeException extends IllegalArgumentException {
        UnsupportedPluginSettingValueTypeException(String s) {
            super(s);
        }

        UnsupportedPluginSettingValueTypeException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    /**
     * @since 8.0
     */
    public static class CorruptPluginSettingValueException extends IllegalStateException {
        CorruptPluginSettingValueException(String s) {
            super(s);
        }

        CorruptPluginSettingValueException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    /**
     * Legacy hook method for storing key-value pairs, used when extending this class via the deprecated default
     * constructor.
     *
     * @param key the setting key (non-null, max 255 characters)
     * @param val the serialized string value to store
     * @throws UnsupportedOperationException if called directly (default implementation)
     * @deprecated since 8.0. Instead of overriding this method, implement {@link KeyValueStore#put(String, String)}
     *             and pass your implementation to
     *             {@link #AbstractStringPluginSettings(KeyValueStore, int, StringSerializationMode)}.
     */
    @Deprecated
    protected void putActual(String key, String val) {
        throw new UnsupportedOperationException();
    }

    /**
     * Legacy hook method for retrieving stored values, used when extending this class via the deprecated default
     * constructor.
     *
     * @param key the setting key to retrieve (non-null, max 255 characters)
     * @return the serialized string value, or {@code null} if not found
     * @throws UnsupportedOperationException if called directly (default implementation)
     * @deprecated since 8.0. Instead of overriding this method, implement {@link KeyValueStore#get(String)}
     *             and pass your implementation to
     *             {@link #AbstractStringPluginSettings(KeyValueStore, int, StringSerializationMode)}.
     */
    @Deprecated
    protected String getActual(String key) {
        throw new UnsupportedOperationException();
    }

    /**
     * Legacy hook method for removing stored values, used when extending this class via the deprecated default
     * constructor.
     *
     * @param key the setting key to remove (non-null, max 255 characters)
     * @throws UnsupportedOperationException if called directly (default implementation)
     * @deprecated since 8.0. Instead of overriding this method, implement {@link KeyValueStore#remove(String)}
     *             and pass your implementation to
     *             {@link #AbstractStringPluginSettings(KeyValueStore, int, StringSerializationMode)}.
     */
    @Deprecated
    protected void removeActual(String key) {
        throw new UnsupportedOperationException();
    }
}
