package com.atlassian.sal.core.net;

import java.net.URI;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.atlassian.plugins.whitelist.OutboundWhitelist;
import com.atlassian.sal.api.net.Request.MethodType;

import static java.util.Objects.requireNonNull;

/**
 * Extended HttpClientRequestFactory which is aware of the outbound whitelist.
 */
public class WhitelistAwareHttpClientRequestFactory extends HttpClientRequestFactory {
    private static final Logger log = LoggerFactory.getLogger(WhitelistAwareHttpClientRequestFactory.class);
    private final OutboundWhitelist outboundWhitelist;

    public WhitelistAwareHttpClientRequestFactory(OutboundWhitelist outboundWhitelist) {
        super();
        this.outboundWhitelist = requireNonNull(outboundWhitelist);
    }

    public WhitelistAwareHttpClientRequestFactory(
            final ProxyConfig proxyConfig, final OutboundWhitelist outboundWhitelist) {
        super(proxyConfig);
        this.outboundWhitelist = requireNonNull(outboundWhitelist);
    }

    @Override
    public HttpClientRequest createRequest(final MethodType methodType, final String url) {
        log.debug("Creating HttpClientRequest with proxy config:", proxyConfigSupplier.get());
        validateUrlAgainstWhitelist(url);
        return super.createRequest(methodType, url);
    }

    private void validateUrlAgainstWhitelist(String url) {
        if (!outboundWhitelist.isAllowed(URI.create(url))) {
            String exceptionMessage = String.format(
                    "The provided url- %s is not included in the whitelist!.Please add the url to whitelist to allow access",
                    url);
            throw new IllegalURLException(exceptionMessage);
        }
    }
}
