package com.atlassian.sal.core.net;

import java.util.function.Supplier;

import org.apache.hc.client5.http.ContextBuilder;
import org.apache.hc.client5.http.auth.AuthCache;
import org.apache.hc.client5.http.auth.AuthScope;
import org.apache.hc.client5.http.auth.CredentialsStore;
import org.apache.hc.client5.http.auth.UsernamePasswordCredentials;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.cookie.StandardCookieSpec;
import org.apache.hc.client5.http.impl.auth.BasicCredentialsProvider;
import org.apache.hc.client5.http.impl.auth.BasicScheme;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.io.HttpClientConnectionManager;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.client5.http.routing.HttpRoutePlanner;
import org.apache.hc.core5.http.HttpHost;
import org.apache.hc.core5.http.impl.io.HttpRequestExecutor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import io.atlassian.util.concurrent.LazyReference;

import com.atlassian.sal.api.net.NonMarshallingRequestFactory;
import com.atlassian.sal.api.net.Request.MethodType;

/**
 * Does NOT support json/xml object marshalling. Use the atlassian-rest implementation of {@link
 * com.atlassian.sal.api.net.RequestFactory} instead.
 */
public class HttpClientRequestFactory implements NonMarshallingRequestFactory<HttpClientRequest<?, ?>> {
    private static final Logger log = LoggerFactory.getLogger(HttpClientRequestFactory.class);

    protected final Supplier<ProxyConfig> proxyConfigSupplier;

    public HttpClientRequestFactory() {
        // must be initialised not earlier than the first use, because
        // system properties for the proxy config could be defined later in the startup
        proxyConfigSupplier = new LazyReference<ProxyConfig>() {
            @Override
            protected ProxyConfig create() {
                return new SystemPropertiesProxyConfig();
            }
        };
    }

    public HttpClientRequestFactory(final ProxyConfig proxyConfig) {
        this.proxyConfigSupplier = () -> proxyConfig;
    }

    /* (non-Javadoc)
     * @see com.atlassian.sal.api.net.RequestFactory#createMethod(com.atlassian.sal.api.net.Request.MethodType, java.lang.String)
     */
    public HttpClientRequest createRequest(final MethodType methodType, final String url) {
        log.debug("Creating HttpClientRequest with proxy config:", proxyConfigSupplier.get());

        final CloseableHttpClient httpClient = createHttpClient();
        final boolean requiresAuthentication = ProxyUtil.requiresAuthentication(proxyConfigSupplier.get(), url);
        final HttpClientContext clientContext = createClientContext(requiresAuthentication);
        return new HttpClientRequest(httpClient, clientContext, methodType, url);
    }

    protected CloseableHttpClient createHttpClient() {
        return HttpClients.custom()
                .useSystemProperties()
                .setRoutePlanner(getRoutePlanner())
                .setRequestExecutor(getRequestExecutor())
                .setConnectionManager(getConnectionManager())
                .setDefaultRequestConfig(RequestConfig.custom()
                        .setCookieSpec(StandardCookieSpec.RELAXED)
                        .build())
                .build();
    }

    protected HttpClientContext createClientContext() {
        return createClientContext(proxyConfigSupplier.get().requiresAuthentication());
    }

    protected HttpClientContext createClientContext(boolean requiresAuthentication) {
        final ContextBuilder contextBuilder = ContextBuilder.create();
        final AuthCache authCache = new AllPortsAuthCache();
        final CredentialsStore basicCredentialsProvider = new BasicCredentialsProvider();
        final ProxyConfig proxyConfig = this.proxyConfigSupplier.get();

        if (requiresAuthentication) {
            HttpHost proxyHost = new HttpHost(proxyConfig.getHost(), proxyConfig.getPort());
            final AuthScope proxyAuthScope = new AuthScope(proxyHost);
            final UsernamePasswordCredentials proxyCredentials = new UsernamePasswordCredentials(
                    proxyConfig.getUser(), proxyConfig.getPassword().toCharArray());
            basicCredentialsProvider.setCredentials(proxyAuthScope, proxyCredentials);
            // This ensures that proxy authentication is preemptive.
            BasicScheme proxyScheme = new BasicScheme();
            proxyScheme.initPreemptive(proxyCredentials);
            authCache.put(proxyHost, proxyScheme);
        }

        contextBuilder.useCredentialsProvider(basicCredentialsProvider);
        contextBuilder.useAuthCache(authCache);
        return contextBuilder.build();
    }

    public boolean supportsHeader() {
        return true;
    }

    protected HttpRoutePlanner getRoutePlanner() {
        return proxyConfigSupplier.get().isSet() ? new ProxyRoutePlanner(proxyConfigSupplier.get()) : null;
    }

    /**
     * We can override the to override the request execution behaviour. This is useful for testing, but potentially
     * also useful in other scenarious.
     *
     * @return HttpRequestExecutor
     */
    protected HttpRequestExecutor getRequestExecutor() {
        return null;
    }

    protected HttpClientConnectionManager getConnectionManager() {
        return null;
    }
}
