import PropTypes from 'prop-types';
import React, { useEffect, useState } from 'react';

import helpPaths from 'help-paths';
import formatter from 'shims/i18n';
import {
    performConnectivityChecksForTheInstance,
    performConnectivityChecksForCDN,
} from 'health-checks/health-checks';
import HealthCheckCard from 'health-checks/health-check-card';
import LearnMore from 'admin-config/learn-more/learn-more';
import ConnectivityResults from './connectivity-results';

const ConnectivityHealthChecks = ({ checks }) => {
    const [results, setResults] = useState(null);

    useEffect(() => {
        Promise.resolve(checks).then(resolvedChecks => setResults(resolvedChecks));
    }, [checks]);

    function renderMessages(healthChecks) {
        const { cdn } = healthChecks;
        const messages = [];

        if (cdn.canServeAssets) {
            messages.push(
                <HealthCheckCard
                    key="cdn.connection.success"
                    appearance="success"
                    title={formatter.I18n.getText(
                        'static.assets.admin.health.cdn.connection.success'
                    )}
                >
                    {formatter.I18n.getText(
                        'static.assets.admin.health.cdn.connection.success.desc'
                    )}
                </HealthCheckCard>
            );

            if (cdn.http2 === false) {
                messages.push(
                    <HealthCheckCard
                        key="cdn.http2.failure"
                        appearance="warning"
                        title={formatter.I18n.getText(
                            'static.assets.admin.health.cdn.http2.failure'
                        )}
                    >
                        {formatter.I18n.getText(
                            'static.assets.admin.health.cdn.http2.failure.desc'
                        )}{' '}
                        <LearnMore url={helpPaths.getHttp2HealthCheckDocURL()} />
                    </HealthCheckCard>
                );
            } else if (cdn.http2 === undefined) {
                messages.push(
                    <HealthCheckCard
                        key="cdn.http2.unknown"
                        appearance="warning"
                        title={formatter.I18n.getText(
                            'static.assets.admin.health.cdn.http2.unknown'
                        )}
                    >
                        {formatter.I18n.getText(
                            'static.assets.admin.health.cdn.http2.unknown.desc'
                        )}{' '}
                        <LearnMore url={helpPaths.getHttp2HealthCheckDocURL()} />
                    </HealthCheckCard>
                );
            }
        } else {
            messages.push(
                <HealthCheckCard
                    key="cdn.connection.failure"
                    appearance="error"
                    title={formatter.I18n.getText(
                        'static.assets.admin.health.cdn.connection.failure'
                    )}
                >
                    {formatter.I18n.getText(
                        'static.assets.admin.health.cdn.connection.failure.desc'
                    )}{' '}
                    <LearnMore url={helpPaths.getNotConnectedHealthCheckDocURL()} />
                </HealthCheckCard>
            );
        }

        return messages;
    }

    return (
        <>
            {results === null ? (
                <section className="loading-container">
                    <aui-spinner size="medium" />
                </section>
            ) : (
                renderMessages(results)
            )}
        </>
    );
};
ConnectivityHealthChecks.propTypes = {
    checks: PropTypes.oneOfType([
        // resolved value
        PropTypes.exact({
            cdn: PropTypes.instanceOf(ConnectivityResults),
            instance: PropTypes.instanceOf(ConnectivityResults),
        }),
        // promise
        PropTypes.objectOf({
            then: PropTypes.func,
            catch: PropTypes.func,
        }),
    ]).isRequired,
};

export default ConnectivityHealthChecks;

export function getConnectivityResults(cdnUrl) {
    return Promise.all([
        performConnectivityChecksForTheInstance(),
        performConnectivityChecksForCDN(cdnUrl),
    ]).then(([instance, cdn]) => {
        return {
            cdn,
            instance,
        };
    });
}
