import React, { useEffect, useState } from 'react';
import Button from '@atlaskit/button';

import helpPaths from 'help-paths';
import formatter from 'shims/i18n';
import { getListOfDeprecatedPlugins } from 'health-checks/deprecated-plugins';
import { PluginList } from 'health-checks/deprecated-plugins.ui.styled';
import HealthCheckCard from 'health-checks/health-check-card';

const DeprecatedPlugins = () => {
    const pluginsInitState = null;
    const [plugins, setPlugins] = useState(pluginsInitState);

    function checkAppsCompatibilityWithCdn() {
        setPlugins(pluginsInitState);
        getListOfDeprecatedPlugins()
            .then(list => setPlugins(list))
            .catch(() => setTimeout(() => setPlugins(false), 1000));
    }

    useEffect(() => {
        checkAppsCompatibilityWithCdn();
    }, []);

    function getLinkToPlugin(plugin) {
        return (
            <li key={plugin.key}>
                <a href={plugin.links.manage}>{plugin.name}</a>
            </li>
        );
    }

    const displayErrorMessage = plugins === false;
    const displayLoader = !displayErrorMessage && plugins === null;
    const displayPlugins = !displayErrorMessage && !displayLoader;

    return (
        <>
            {displayPlugins &&
                (plugins.length ? (
                    <HealthCheckCard
                        appearance="info"
                        title={formatter.I18n.getText(
                            'static.assets.admin.health.deprecated.plugin'
                        )}
                    >
                        {formatter.I18n.getText(
                            'static.assets.admin.health.deprecated.plugin.desc'
                        )}
                        <PluginList>
                            <ul>{plugins.map(plugin => getLinkToPlugin(plugin))}</ul>
                        </PluginList>
                        <div
                            /* eslint-disable-next-line react/no-danger */
                            dangerouslySetInnerHTML={{
                                __html: helpPaths.getAppsHealthCheckLearnMore(),
                            }}
                        />
                    </HealthCheckCard>
                ) : (
                    <HealthCheckCard
                        appearance="success"
                        title={formatter.I18n.getText(
                            'static.assets.admin.health.deprecated.plugin.empty.state'
                        )}
                    >
                        {formatter.I18n.getText(
                            'static.assets.admin.health.deprecated.plugin.empty.state.desc'
                        )}
                    </HealthCheckCard>
                ))}
            {displayLoader && (
                <section className="loading-container">
                    <aui-spinner size="medium" />
                </section>
            )}
            {displayErrorMessage && (
                <HealthCheckCard
                    appearance="error"
                    title={formatter.I18n.getText(
                        'static.assets.admin.health.deprecated.plugins.failure'
                    )}
                >
                    <Button
                        appearance="link"
                        spacing="none"
                        onClick={checkAppsCompatibilityWithCdn}
                    >
                        {formatter.I18n.getText(
                            'static.assets.admin.health.deprecated.plugins.failure.action'
                        )}
                    </Button>
                </HealthCheckCard>
            )}
        </>
    );
};

export default DeprecatedPlugins;
