import { getCdnSettings, saveCdnSettings } from 'admin-config/cdn-data-service';
import CdnData from 'admin-config/cdn-data';
import { mockFetchFailure, mockFetchSuccess, restoreFetch } from '../fetch';

describe('getCdnSettings', () => {
    afterEach(() => restoreFetch);

    it('Should call proper URL', async () => {
        const fetchMock = mockFetchSuccess(new CdnData());

        await getCdnSettings();

        expect(fetchMock)
            .toHaveBeenCalledTimes(1)
            .toBeCalledWith('/context/rest/static-asset-caching/configuration');
    });

    {
        const testData = [
            new CdnData(),
            new CdnData(undefined, undefined),
            new CdnData(true, 'test'),
            new CdnData(null, null),
            new CdnData(false, ''),
        ];
        test.each`
            mockData       | returnData     | desc
            ${testData[0]} | ${testData[0]} | ${'#1: default values of CdnData'}
            ${testData[1]} | ${testData[1]} | ${'#2: same as #1 but with `undefined`'}
            ${testData[2]} | ${testData[2]} | ${'#3: expected data'}
            ${testData[3]} | ${testData[4]} | ${'#4: should handle `null` values'}
        `(
            'Should resolve with correct data on success ($desc)',
            async ({ mockData, returnData }) => {
                mockFetchSuccess(mockData);

                const cdnData = await getCdnSettings();

                expect(cdnData).toBeInstanceOf(CdnData);
                expect(cdnData).toEqual(returnData);
            }
        );
    }

    test.each`
        returnData
        ${null}
        ${{}}
        ${false}
        ${''}
        ${{ url: '' }}
        ${{ enabled: true }}
    `(
        'Should reject with error when server responded with malformed data: $returnData',
        async ({ returnData }) => {
            expect.assertions(2);
            mockFetchSuccess(returnData);

            await getCdnSettings().catch(err => {
                expect(err).toBeInstanceOf(Error);
                expect(err.message).toEqual('Malformed CDN data received');
            });
        }
    );

    it('Should reject with error on server failure', done => {
        mockFetchSuccess(null, false);

        getCdnSettings().catch(err => {
            expect(err).toBeInstanceOf(Error);
            expect(err.message).toEqual('Fetch failed');
            done();
        });
    });

    it('Should reject on fetch failure', done => {
        mockFetchFailure(new TypeError('Failed to fetch'));

        getCdnSettings().catch(err => {
            expect(err).toBeInstanceOf(TypeError);
            expect(err.message).toEqual('Failed to fetch');
            done();
        });
    });
});

describe('saveCdnSettings', () => {
    afterEach(() => restoreFetch);

    it('Should call proper URL and params', async () => {
        const newData = new CdnData(true, 'https://cdn.example.com');
        const fetchMock = mockFetchSuccess(new CdnData());

        await saveCdnSettings(newData);

        expect(fetchMock)
            .toHaveBeenCalledTimes(1)
            .toBeCalledWith('/context/rest/static-asset-caching/configuration', {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(newData),
            });
    });

    {
        const testData = [
            new CdnData(true, 'https://cdn.example.com'),
            new CdnData(false, ''),
            new CdnData(undefined, undefined),
        ];
        test.each`
            newData        | returnData     | desc
            ${testData[0]} | ${testData[0]} | ${'new settings are reflected in the server response'}
            ${testData[1]} | ${testData[1]} | ${'new settings are reflected in the server response #2'}
            ${testData[0]} | ${testData[1]} | ${'server may return different settings'}
            ${testData[1]} | ${testData[0]} | ${'server may return different settings #2'}
            ${testData[2]} | ${testData[1]} | ${'edge case #1'}
        `(
            'Should resolve with correct data on success ($desc)',
            async ({ newData, returnData }) => {
                mockFetchSuccess(returnData);

                const cdnData = await saveCdnSettings(newData);

                expect(cdnData).toBeInstanceOf(CdnData);
                expect(cdnData).toEqual(returnData);
            }
        );
    }

    it('Should reject with error when sent invalid data to the server', async () => {
        expect.assertions(2);
        mockFetchSuccess(null, false);

        await saveCdnSettings(new CdnData(true, '')).catch(err => {
            expect(err).toBeInstanceOf(Error);
            expect(err.message).toEqual('Fetch failed');
        });
    });

    it('Should reject with error on server failure', done => {
        mockFetchSuccess(null, false);

        saveCdnSettings().catch(err => {
            expect(err).toBeInstanceOf(Error);
            expect(err.message).toEqual('Fetch failed');
            done();
        });
    });

    it('Should reject on fetch failure', done => {
        mockFetchFailure(new TypeError('Failed to fetch'));

        saveCdnSettings().catch(err => {
            expect(err).toBeInstanceOf(TypeError);
            expect(err.message).toEqual('Failed to fetch');
            done();
        });
    });
});
