package com.atlassian.plugins.rest.v2.security.authentication;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import jakarta.annotation.Priority;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.Priorities;
import jakarta.ws.rs.container.ContainerRequestContext;
import jakarta.ws.rs.container.ContainerRequestFilter;
import jakarta.ws.rs.container.ResourceInfo;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.ext.Provider;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.atlassian.annotations.security.ScopesAllowed;
import com.atlassian.oauth2.scopes.api.ScopesRequestCache;
import com.atlassian.plugins.rest.api.security.annotation.AnonymousSiteAccess;
import com.atlassian.plugins.rest.api.security.exception.AuthenticationRequiredException;
import com.atlassian.plugins.rest.api.security.exception.AuthorizationException;
import com.atlassian.sal.api.features.DarkFeatureManager;
import com.atlassian.sal.api.user.UserKey;
import com.atlassian.sal.api.user.UserManager;

/**
 * This is a Container Request Filter that checks whether the current client has access to current resource or its method. If the
 * client doesn't  have access then an {@link AuthenticationRequiredException} is thrown.
 *
 * <p>
 * Resources can be marked as not needing authentication by using the
 * {@link AnonymousSiteAccess} annotation
 */
@Priority(Priorities.AUTHENTICATION)
@Provider
public class AuthenticatedResourceFilter implements ContainerRequestFilter {

    private static final Logger log = LoggerFactory.getLogger(AuthenticatedResourceFilter.class);

    public static final String DEFAULT_TO_LICENSED_ACCESS_DISABLED_FEATURE_KEY =
            "atlassian.rest.default.to.licensed.access.disabled";
    public static final String DEFAULT_TO_SYSADMIN_ACCESS_FEATURE_KEY =
            "atlassian.rest.default.to.sysadmin.access.enabled";

    private final UserManager userManager;
    private final DarkFeatureManager darkFeatureManager;
    private final ScopesRequestCache scopesRequestCache;

    @Context
    private ResourceInfo resourceInfo;

    @Context
    private HttpServletRequest request;

    public AuthenticatedResourceFilter(
            UserManager userManager, DarkFeatureManager darkFeatureManager, ScopesRequestCache scopesRequestCache) {
        this.userManager = userManager;
        this.darkFeatureManager = darkFeatureManager;
        this.scopesRequestCache = scopesRequestCache;
    }

    @Override
    public void filter(ContainerRequestContext requestContext) {
        UserKey userKey = userManager.getRemoteUserKey();

        boolean isAnonymous = userKey == null;
        boolean isOAuth2 = request.getAttribute("oauth2.token.client_configuration_id") != null;
        boolean isServiceAccount = request.getAttribute("serviceAccount") != null;
        boolean is2LO = isOAuth2 && (isServiceAccount || isAnonymous);
        boolean is3LO = isOAuth2 && !is2LO;

        if (!isOAuth2) {
            // For non-OAuth2 requests, we check if the request has sufficient security annotations
            requireAccessGrantedByAccessType(userKey);
        } else if (is3LO) {
            // For 3LO requests, we check if the request has sufficient security annotations
            requireAccessGrantedByAccessType(userKey);

            requireAccessGrantedByOAuth2Scopes_3LO();
        } else {
            // OAuth2 2LO case
            // We do not check security annotations for 2LO requests.
            // Scopes are required for 2LO requests.
            // 2 cases here:
            // First case - a service account is not subject to security annotations as a service account
            // will always be a licensed user and CAN NEVER be an admin/sysadmin.
            // We decided that we don't want to check license for service accounts for every request.
            // Second case - anonymous access is allowed for 2LO requests (scopes required).
            requireAccessGrantedByOAuth2Scopes_2LO();
        }
    }

    void requireAccessGrantedByAccessType(UserKey userKey) {
        AccessType requiredAccessType = getRequiredAccessType();
        boolean isGranted = grantAccessBySecurityAnnotations(requiredAccessType);
        if (!isGranted) {
            throw mapToException(userKey, requiredAccessType);
        }
    }

    void requireAccessGrantedByOAuth2Scopes_3LO() {
        String[] resourceScopes = getScopesRequiredForResource(resourceInfo);
        Set<String> scopeApplicableFor3LO = Arrays.stream(resourceScopes)
                .filter(scopesRequestCache::isScopeApplicableFor3LO)
                .collect(Collectors.toSet());

        // methods without any @ScopesAllowed allow 3LO
        if (resourceScopes.length > 0) {
            // any method with a @ScopesAllowed annotation must have a 3LO-valid scope or it can’t be used by 3LO
            if (scopeApplicableFor3LO.isEmpty()) {
                throw new AuthorizationException("This resource is not allowed for 3LO access.");
            }

            // Will check if this request has credentials/access token that contains grants for ANY the given scopes.
            boolean isOAuth2ScopesGranted = scopesRequestCache.requestHasGrantsForAny(scopeApplicableFor3LO);

            if (!isOAuth2ScopesGranted) {
                throw mapToException(resourceScopes);
            }
        }
    }

    void requireAccessGrantedByOAuth2Scopes_2LO() {
        String[] resourceScopes = getScopesRequiredForResource(resourceInfo);
        Set<String> scopeApplicableFor2LO = Arrays.stream(resourceScopes)
                .filter(scopesRequestCache::isScopeApplicableFor2LO)
                .collect(Collectors.toSet());

        // Scopes are required for 2LO requests.
        if (scopeApplicableFor2LO.isEmpty()) {
            throw new AuthorizationException("This resource is not allowed for 2LO access.");
        }

        boolean isOAuth2ScopesGranted = scopesRequestCache.requestHasGrantsForAny(scopeApplicableFor2LO);
        if (!isOAuth2ScopesGranted) {
            throw mapToException(resourceScopes);
        }
    }

    private boolean grantAccessBySecurityAnnotations(AccessType requiredAccessType) {
        final UserKey userKey = userManager.getRemoteUserKey();

        return switch (requiredAccessType) {
            case SYSTEM_ADMIN_ONLY -> userManager.isSystemAdmin(userKey);

            case ADMIN_ONLY -> userManager.isAdmin(userKey) || userManager.isSystemAdmin(userKey);

            case LICENSED_ONLY -> userManager.isLicensed(userKey);

            case UNLICENSED_SITE_ACCESS -> userManager.isLicensed(userKey)
                    || userManager.isLimitedUnlicensedUser(userKey);

            case ANONYMOUS_SITE_ACCESS -> {
                boolean isAnonymous = userKey == null;
                boolean isAnonymousAllowed = isAnonymous && userManager.isAnonymousAccessEnabled();
                boolean isLicensedAllowed =
                        userManager.isLicensed(userKey) || userManager.isLimitedUnlicensedUser(userKey);

                yield isAnonymousAllowed || isLicensedAllowed;
            }

            default -> AccessType.UNRESTRICTED_ACCESS == requiredAccessType;
        };
    }

    private AccessType getRequiredAccessType() {
        AccessType accessType =
                AccessType.getAccessType(resourceInfo.getResourceClass(), resourceInfo.getResourceMethod());
        if (AccessType.EMPTY == accessType) {
            // DragonFly feature flag - unannotated resources should default to system admin access
            if (darkFeatureManager
                    .isEnabledForAllUsers(DEFAULT_TO_SYSADMIN_ACCESS_FEATURE_KEY)
                    .orElse(false)) {
                accessType = AccessType.SYSTEM_ADMIN_ONLY;
            } else if (darkFeatureManager
                    .isEnabledForAllUsers(DEFAULT_TO_LICENSED_ACCESS_DISABLED_FEATURE_KEY)
                    .orElse(false)) {
                accessType = AccessType.UNRESTRICTED_ACCESS;
            } else {
                accessType = AccessType.LICENSED_ONLY;
            }
        }
        return accessType;
    }

    private static String[] getScopesRequiredForResource(ResourceInfo resourceInfo) {
        Method resourceMethod = resourceInfo.getResourceMethod();
        Optional<ScopesAllowed> scopesAllowedAnnotation = Stream.of(resourceMethod)
                .filter(Objects::nonNull)
                .map(annotatedElement -> annotatedElement.getAnnotation(ScopesAllowed.class))
                .filter(Objects::nonNull)
                .findFirst();
        if (scopesAllowedAnnotation.isPresent()) {
            return scopesAllowedAnnotation.get().requiredScope();
        }

        // Check if the annotation is present but is from a different class loader, log an error
        Optional<Annotation> annotationFromDifferentClassLoader = Stream.of(resourceMethod)
                .filter(Objects::nonNull)
                .flatMap(annotatedElement -> Arrays.stream(annotatedElement.getAnnotations()))
                .filter(annotation -> annotation.annotationType().getName().equals(ScopesAllowed.class.getName()))
                .findFirst();
        if (annotationFromDifferentClassLoader.isPresent()) {
            Class<?> resourceClass = resourceInfo.getResourceClass();
            log.error(
                    "Resource class={} method={} has ScopesAllowed annotation from a different class loader, "
                            + "ignoring it. Please ensure that atlassian-annotations dependency is added with a "
                            + "provided scope in the plugin's pom.xml",
                    resourceClass,
                    resourceMethod);
        }
        return new String[0];
    }

    private static SecurityException mapToException(String[] requiredScopes) {
        String message = String.format(
                """
                Client must have sufficient scopes to access this resource. Required scopes: %s""",
                Arrays.toString(requiredScopes));
        log.debug(message);
        return new AuthorizationException(message);
    }

    private static SecurityException mapToException(UserKey userKey, AccessType accessType) {
        if (userKey == null) {
            return new AuthenticationRequiredException();
        }
        if (accessType == AccessType.SYSTEM_ADMIN_ONLY) {
            return new AuthorizationException(
                    "Client must be authenticated as a system administrator to access this resource.");
        }
        if (accessType == AccessType.ADMIN_ONLY) {
            return new AuthorizationException(
                    "Client must be authenticated as an administrator to access this resource.");
        }
        if (accessType == AccessType.LICENSED_ONLY) {
            throw new AuthorizationException(
                    "Client must be authenticated as a licensed user to access this resource.");
        }
        return new AuthorizationException("Client must have sufficient permissions to access this resource.");
    }
}
