package com.atlassian.plugins.rest.common.security.jersey;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.plugins.rest.common.security.AdminOnly;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;
import com.atlassian.plugins.rest.common.security.AnonymousSiteAccess;
import com.atlassian.plugins.rest.common.security.AuthenticationRequiredException;
import com.atlassian.plugins.rest.common.security.LicensedOnly;
import com.atlassian.plugins.rest.common.security.SystemAdminOnly;
import com.atlassian.plugins.rest.common.security.UnlicensedSiteAccess;
import com.atlassian.plugins.rest.common.security.UnrestrictedAccess;
import com.atlassian.plugins.rest.common.util.AnnotationUtils;
import com.atlassian.sal.api.features.DarkFeatureManager;
import com.atlassian.sal.api.user.UserKey;
import com.atlassian.sal.api.user.UserManager;
import com.sun.jersey.spi.container.ContainerRequest;
import com.sun.jersey.spi.container.ContainerRequestFilter;
import com.sun.jersey.spi.container.ContainerResponseFilter;
import com.sun.jersey.spi.container.ResourceFilter;

import java.lang.annotation.Annotation;

import static java.util.Objects.requireNonNull;

/**
 * This is a Jersey resource filter that checks whether the current client has access to current resource or its method. If the client doesn't
 * have access then an {@link AuthenticationRequiredException} is thrown.
 *
 * <p>
 * Resources can be marked as not needing authentication by using the {@link com.atlassian.plugins.rest.common.security.AnonymousSiteAccess} annotation
 */
class AuthenticatedResourceFilter implements ResourceFilter, ContainerRequestFilter {
    @VisibleForTesting
    static final String DEFAULT_TO_LICENSED_ACCESS_FEATURE_KEY = "atlassian.rest.default.to.licensed.access.enabled";
    private final AnnotationUtils annotationUtils;
    private final UserManager userManager;
    private final DarkFeatureManager darkFeatureManager;

    public AuthenticatedResourceFilter(final AnnotationUtils annotationUtils, UserManager userManager, final DarkFeatureManager darkFeatureManager) {
        this.annotationUtils = requireNonNull(annotationUtils, "annotationUtils can't be null");
        this.userManager = requireNonNull(userManager, "userManager can't be null");
        this.darkFeatureManager = requireNonNull(darkFeatureManager, "featureFlagManager can't be null");
    }

    public ContainerRequestFilter getRequestFilter() {
        return this;
    }

    public ContainerResponseFilter getResponseFilter() {
        return null;
    }

    public ContainerRequest filter(ContainerRequest request) {
        Class<Annotation> annotation = annotationUtils.getAnnotation();
        UserKey userKey = userManager.getRemoteUserKey();

        if(annotation == null) {
            boolean defaultToLicensedResourceAccessFlag = darkFeatureManager.isFeatureEnabledForAllUsers(DEFAULT_TO_LICENSED_ACCESS_FEATURE_KEY);

            if ((defaultToLicensedResourceAccessFlag && userKey != null && userManager.isLicensed(userKey)) ||
                    (!defaultToLicensedResourceAccessFlag && userKey != null)) {
                return request;
            }
        } else if (SystemAdminOnly.class.equals(annotation)) {
            if (userKey != null && userManager.isSystemAdmin(userKey)) {
                return request;
            }
        } else if (AdminOnly.class.equals(annotation)) {
            if (userKey != null && (userManager.isSystemAdmin(userKey) || userManager.isAdmin(userKey))) {
                return request;
            }
        } else if (LicensedOnly.class.equals(annotation)) {
            if (userKey != null && userManager.isLicensed(userKey)) {
                return request;
            }
        } else if (UnlicensedSiteAccess.class.equals(annotation)) {
            if (userKey != null && (userManager.isLicensed(userKey) || userManager.isLimitedUnlicensedUser(userKey))) {
                return request;
            }
        } else if (AnonymousSiteAccess.class.equals(annotation)) {
            if ((userKey == null && userManager.isAnonymousAccessEnabled()) || (userKey != null && (userManager.isLicensed(userKey) || userManager.isLimitedUnlicensedUser(userKey)))) {
                return request;
            }
        } else if (UnrestrictedAccess.class.equals(annotation)) {
            return request;
        } else if (AnonymousAllowed.class.equals(annotation)) {
            return request;
        }

        throw new AuthenticationRequiredException();
    }
}
