package com.atlassian.plugins.rest.common.security.jersey;

import com.atlassian.annotations.security.XsrfProtectionExcluded;
import com.atlassian.plugins.rest.common.security.RequiresXsrfCheck;
import com.atlassian.sal.api.web.context.HttpContext;
import com.atlassian.sal.api.xsrf.XsrfTokenValidator;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.sun.jersey.api.model.AbstractMethod;
import com.sun.jersey.spi.container.ResourceFilter;
import com.sun.jersey.spi.container.ResourceFilterFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.ext.Provider;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.util.Collections;
import java.util.List;

/**
 * Factory for the XSRF resource filter
 *
 * @since 2.4
 */
@Provider
public class XsrfResourceFilterFactory implements ResourceFilterFactory
{
    private static final String XSRF_ANNOTATION_CLASS_NAME =
        RequiresXsrfCheck.class.getSimpleName();
    private static final Logger log = LoggerFactory.getLogger(
        XsrfResourceFilterFactory.class);

    private HttpContext httpContext;
    private XsrfTokenValidator xsrfTokenValidator;


    public XsrfResourceFilterFactory(HttpContext httpContext, XsrfTokenValidator xsrfTokenValidator)
    {
        this.httpContext = Preconditions.checkNotNull(httpContext);
        this.xsrfTokenValidator = Preconditions.checkNotNull(xsrfTokenValidator);
    }

    @VisibleForTesting
    boolean hasRequiresXsrfCheckAnnotation(AnnotatedElement annotatedElement)
    {
        if (annotatedElement.isAnnotationPresent(RequiresXsrfCheck.class))
        {
            return true;
        }
        for (Annotation annotation : annotatedElement.getAnnotations())
        {
            if (annotation.annotationType().getSimpleName().equals(
                XSRF_ANNOTATION_CLASS_NAME))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns true if any of the given annotations is the
     * XsrfProtectionExcluded annotation.
     * @param annotations the annotations to check against.
     * @return true if any of the given annotations is the
     * XsrfProtectionExcluded annotation. Otherwise returns false.
     *
     */
    private static boolean isXsrfProtectionExcludedAnnotationPresent(
        Annotation[] annotations)
    {
        for (Annotation annotation: annotations)
        {
            if (annotation.annotationType().getCanonicalName().equals(
                XsrfProtectionExcluded.class.getCanonicalName()))
            {
                if (!annotation.annotationType().equals(
                    XsrfProtectionExcluded.class))
                {
                    log.warn("Detected usage of the " +
                        "com.atlassian.annotations.security." +
                        "XsrfProtectionExcluded annotation loaded " +
                        "from elsewhere. " +
                        XsrfProtectionExcluded.class.getClassLoader() +
                        " != " + annotation.annotationType().getClassLoader()
                    );
                }
                return true;
            }
        }
        return false;
    }

    public List<ResourceFilter> create(final AbstractMethod method)
    {
        if (!isXsrfProtectionExcludedAnnotationPresent(method.getAnnotations())
            && (hasRequiresXsrfCheckAnnotation(method) ||
            hasRequiresXsrfCheckAnnotation(method.getResource())))
        {
            XsrfResourceFilter xsrfResourceFilter = new XsrfResourceFilter();
            xsrfResourceFilter.setHttpContext(httpContext);
            xsrfResourceFilter.setXsrfTokenValidator(xsrfTokenValidator);
            return Collections.<ResourceFilter>singletonList(xsrfResourceFilter);
        }
        return Collections.emptyList();
    }
}
