package com.atlassian.plugins.rest.common.expand.resolver;

import com.google.common.base.Preconditions;
import com.atlassian.plugins.rest.common.expand.EntityExpander;
import com.atlassian.plugins.rest.common.expand.Expander;
import com.atlassian.plugins.rest.common.expand.ExpandException;

/**
 * <p>An {@link EntityExpander} resolver that uses the {@link Expander} annotation for resolution.</p>
 * <p>Implementation must implement {@link #getEntityExpander(Expander)}</p>
 */
public abstract class AbstractAnnotationEntityExpanderResolver implements EntityExpanderResolver
{
    /**
     * {@inheritDoc}
     */
    public final <T> EntityExpander<T> getExpander(T instance)
    {
        return (EntityExpander<T>) getExpander(Preconditions.checkNotNull(instance).getClass());
    }

    /**
     * {@inheritDoc}
     */
    public final <T> EntityExpander<T> getExpander(Class<? extends T> type)
    {
        final Expander expander = Preconditions.checkNotNull(type).getAnnotation(Expander.class);
        if (expander == null)
        {
            throw new ExpandException("Could not find expander for type <" + type + ">. No @Expander annotation defined on type.");
        }

        return (EntityExpander<T>) getEntityExpander(expander);
    }

    /**
     * Retrieves the {@link EntityExpander} associated to the {@link Expander} annotation. The entity expander is created if necessary.
     * @param expander the annotation
     * @return an instance of {@link EntityExpander}
     */
    protected abstract EntityExpander<?> getEntityExpander(Expander expander);
}
