package com.atlassian.plugins.rest.common.expand;

import com.atlassian.plugins.rest.common.expand.parameter.ExpandParameter;
import com.atlassian.plugins.rest.common.expand.resolver.EntityExpanderResolver;
import static com.atlassian.plugins.rest.common.util.ReflectionUtils.*;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.Collection;

/**
 * This allows for crawling the fields of any arbitrary object, looking for fields that should be expanded.
 */
public final class EntityCrawler
{
    /**
     * Crawls an entity for fields that should be expanded and expands them.
     * @param entity the object to crawl, can be {@code null}.
     * @param expandParameter the parameters to match for expansion
     * @param expanderResolver the resolver to lookup {@link EntityExpander} for fields to be expanded.
     */
    public void crawl(Object entity, ExpandParameter expandParameter, EntityExpanderResolver expanderResolver)
    {
        if (entity == null)
        {
            return;
        }

        final Collection<Field> expandableFields = getExpandableFields(entity);
        setExpandParameter(expandableFields, entity);
        expandFields(expandableFields, entity, expandParameter, expanderResolver);
    }

    private void setExpandParameter(Collection<Field> expandableFields, Object entity)
    {
        final Field expand = getExpandField(entity);
        if (expand != null && !expandableFields.isEmpty())
        {
            final StringBuilder expandValue = new StringBuilder();
            for (Field field : expandableFields)
            {
                expandValue.append(getExpandable(field).value()).append(",");
            }
            expandValue.deleteCharAt(expandValue.length() - 1); // remove the last ","

            setFieldValue(expand, entity, expandValue.toString());
        }
    }

    private Field getExpandField(Object entity)
    {
        for (Field field : entity.getClass().getDeclaredFields())
        {
            if (field.getType().equals(String.class))
            {
                final XmlAttribute annotation = field.getAnnotation(XmlAttribute.class);
                if (annotation != null && (field.getName().equals("expand") || "expand".equals(annotation.name())))
                {
                    return field;
                }
            }
        }
        return null;
    }

    private Collection<Field> getExpandableFields(Object entity)
    {
        final Collection<Field> expandableFields = Lists.newLinkedList();
        for (Field field : entity.getClass().getDeclaredFields())
        {
            if (getExpandable(field) != null)
            {
                expandableFields.add(field);
            }
        }
        return expandableFields;
    }

    private void expandFields(Collection<Field> expandableFields, Object entity, ExpandParameter expandParameter, EntityExpanderResolver expanderResolver)
    {
        for (Field field : expandableFields)
        {
            final Expandable expandable = getExpandable(field);
            if (expandParameter.shouldExpand(expandable))
            {
                final EntityExpander<Object> entityExpander = expanderResolver.getExpander(field.getType());
                final ExpandContext<Object> context = new DefaultExpandContext<Object>(getFieldValue(field, entity), expandable, expandParameter);
                setFieldValue(field, entity, entityExpander.expand(context, expanderResolver, this));
            }
        }
    }

    private Expandable getExpandable(final Field field)
    {
        final Expandable expandable = field.getAnnotation(Expandable.class);

        if (expandable == null)
        {
            return null;
        }

        if (StringUtils.isNotEmpty(expandable.value()))
        {
            return expandable;
        }

        final String xmlElementName = field.getAnnotation(XmlElement.class).name();
        if (StringUtils.isNotEmpty(xmlElementName) && !StringUtils.equals("##default", xmlElementName))
        {
            return new ExpandableWithValue(xmlElementName);
        }

        return new ExpandableWithValue(field.getName());
    }

    private static class ExpandableWithValue implements Expandable
    {
        private final String value;

        public ExpandableWithValue(String value)
        {
            this.value = value;
        }

        public String value()
        {
            return value;
        }

        public Class<? extends Annotation> annotationType()
        {
            return Expandable.class;
        }
    }
}
