package com.atlassian.plugins.less;

import com.atlassian.lesscss.Constants;
import com.atlassian.lesscss.spi.DimensionAwareUriResolver;
import com.atlassian.lesscss.spi.EncodeStateResult;
import com.atlassian.plugin.servlet.ServletContextFactory;
import com.atlassian.webresource.api.prebake.Coordinate;
import com.atlassian.webresource.api.prebake.Dimensions;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.util.Optional;

public class WebStaticUriResolver implements DimensionAwareUriResolver {

    private final ServletContextFactory servletContextFactory;

    public WebStaticUriResolver(ServletContextFactory servletContextFactory) {
        this.servletContextFactory = servletContextFactory;
    }

    @Override
    public Dimensions computeDimensions() {
        return Dimensions.empty();
    }

    @Override
    public boolean exists(URI uri) {
        try {
            return servletContextFactory.getServletContext().getResource(uri.getPath()) != null;
        } catch (MalformedURLException e) {
            return false;
        }
    }

    @Override
    public EncodeStateResult encodeState(URI uri, Coordinate coord) {
        return new EncodeStateResult(encodeState(uri), Optional.empty());
    }

    @Override
    public String encodeState(URI uri) {
        URLConnection connection = null;
        try {
            URL url = servletContextFactory.getServletContext().getResource(uri.getPath());
            connection = url.openConnection();
            return String.valueOf(connection.getLastModified());
        } catch (IOException e) {
            throw new IllegalStateException(e);
        } finally {
            if (connection != null) {
                try {
                    connection.getInputStream().close();
                } catch (IOException ignored) {
                }
            }
        }
    }

    @Override
    public InputStream open(URI uri) throws IOException {
        InputStream is = servletContextFactory.getServletContext().getResourceAsStream(uri.getPath());

        if (is == null) {
            throw new IOException(uri.getPath() + " does not exist in the servletContext");
        }

        return is;
    }

    @Override
    public boolean supports(URI uri) {
        return Constants.SCHEME_WEB_STATIC.equals(uri.getScheme());
    }
}
