package com.atlassian.plugins.less;

import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.lesscss.spi.UriResolver;
import com.atlassian.lesscss.spi.UriResolverStateChangedEvent;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.io.LineReader;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URI;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class CachingUriStateManager implements UriStateManager {

    private static final Pattern IMPORT_URI_PATTERN = Pattern.compile("" +
            "@import(?:-once)?\\s+" + // import statement
            "(?:\\(([^)]+)\\)\\s+)?" + // import options
            "\"([^\"]+)\";"); // import uri

    private static final int GROUP_IMPORT_OPTIONS = 1;
    private static final int GROUP_IMPORT_URI = 2;

    private static final Logger log = LoggerFactory.getLogger(CachingUriStateManager.class);

    private final LoadingCache<URI, UriInfo> cache;
    private final EventPublisher eventPublisher;
    private final UriResolverManager uriResolverManager;


    public CachingUriStateManager(EventPublisher eventPublisher, UriResolverManager uriResolverManager) {
        this.eventPublisher = eventPublisher;
        this.uriResolverManager = uriResolverManager;

        this.cache = CacheBuilder.newBuilder()
                .build(new CacheLoader<URI, UriInfo>() {
                    @Override
                    public UriInfo load(URI uri) throws Exception {
                        return computeUriInfo(uri);
                    }
                });
    }

    public void registerEventListeners() throws Exception {
        eventPublisher.register(this);
    }

    public void unRegisterEventListeners() throws Exception {
        eventPublisher.unregister(this);
    }

    @Override
    public String getState(URI uri) {
        List<String> states = Lists.newArrayList();
        collectUriState(Sets.newHashSet(uri), states, uri, true);

        return Joiner.on(',').join(states);
    }

    @EventListener
    public void onStateChanged(UriResolverStateChangedEvent event) {
        for (Iterator<Map.Entry<URI, UriInfo>> it = cache.asMap().entrySet().iterator(); it.hasNext(); ) {
            Map.Entry<URI, UriInfo> entry = it.next();
            if (event.hasChanged(entry.getKey())) {
                log.debug("LESS has changed. Expiring lastModified cache. uri={}", entry.getKey());
                it.remove();
            }
        }
    }

    @VisibleForTesting
    void collectUris(Set<URI> collector, URI baseUri, String chunk) {
        Matcher matcher = IMPORT_URI_PATTERN.matcher(chunk);
        while (matcher.find()) {
            ImportOption importOption = ImportOption.fromString(matcher.group(GROUP_IMPORT_OPTIONS));
            String path = matcher.group(GROUP_IMPORT_URI);
            if (path != null) {
                path = importOption.amendURI(path);
                URI uri = baseUri.resolve(path);
                if (isUriSupported(uri)) {
                    collector.add(uri);
                } else {
                    log.warn("Ignoring LESS uri as it is not supported. uri={}", uri);
                }
            }
        }
    }

    private void collectUriState(Set<URI> alreadySeen, List<String> states, URI uri, boolean root) {
        UriInfo value = cache.getUnchecked(uri);
        if (root && value.preCompiledState != null) {
            // If the pre-compiled version is being used, the dependencies will never be
            // used to produce the contents so we short-circuit the method. However we
            // Only do this for the 'root' URI. If we are traversing a dependencies state
            // we will be using the un-compiled version.
            states.add(value.preCompiledState);
            return;
        }

        states.add(value.state);

        for (URI dependency : value.dependencies) {
            if (alreadySeen.add(dependency)) {
                collectUriState(alreadySeen, states, dependency, false);
            }
        }
    }

    private UriInfo computeUriInfo(URI uri) {
        log.debug("Computing LESS uri info. uri={}", uri);
        UriResolver uriResolver = getResolverOrThrow(uri);
        URI preCompiledUri = PreCompilationUtils.resolvePreCompiledUri(uriResolver, uri);
        Set<URI> dependencies = getDependencies(uri);

        return new UriInfo(
                dependencies,
                preCompiledUri == null ? null : uriResolver.encodeState(preCompiledUri),
                uriResolver.encodeState(uri)
        );
    }

    /**
     * extract the import dependencies for the supplied uri.
     * <p/>
     * Its horrible that we have to do this. Unfortunately this is the only way to accurately determine
     * when a LESS resource changes
     */
    private Set<URI> getDependencies(final URI baseUri) {
        final Set<URI> collector = Sets.newLinkedHashSet(); // Use a linked hash set so that the iteration is stable
        try (Reader reader = new InputStreamReader(getResolverOrThrow(baseUri).open(baseUri))) {
            LineReader lineReader = new LineReader(reader);
            String line;
            while ((line = lineReader.readLine()) != null) {
                collectUris(collector, baseUri, line);
            }
        } catch (IOException e) {
            throw new IllegalStateException(e);
        }
        return Collections.unmodifiableSet(collector);
    }

    private UriResolver getResolverOrThrow(URI uri) {
        UriResolver uriResolver = getResolver(uri);
        if (uriResolver == null) {
            throw new IllegalArgumentException("Unsupported URI " + uri.toASCIIString());
        }
        return uriResolver;
    }

    private UriResolver getResolver(URI uri) {
        for (UriResolver uriResolver : uriResolverManager.getResolvers()) {
            if (uriResolver.supports(uri)) {
                return uriResolver;
            }
        }
        return null;
    }

    private boolean isUriSupported(URI uri) {
        return getResolver(uri) != null;
    }

    private static enum ImportOption {
        CSS,
        INLINE {
            @Override
            String amendURI(String uri) {
                if (!uri.endsWith(".css")) {
                    uri += ".css";
                }
                return uri;
            }
        },
        LESS,
        NONE {
            @Override
            String amendURI(String uri) {
                if (!uri.endsWith(".less")) {
                    uri += ".less";
                }
                return uri;
            }
        };

        String amendURI(String uri) {
            return uri;
        }

        private static ImportOption fromString(String value) {
            if (value != null) {
                value = value.toUpperCase(Locale.US);
                for (ImportOption importOption : values()) {
                    if (importOption.name().equals(value)) {
                        return importOption;
                    }
                }
            }
            return NONE;
        }
    }

    private static class UriInfo {

        private final Set<URI> dependencies;
        private final String preCompiledState;
        private final String state;

        private UriInfo(Set<URI> dependencies, String preCompiledState, String state) {
            this.dependencies = dependencies;
            this.preCompiledState = preCompiledState;
            this.state = state;
        }

    }


}
