AJS.namespace("JIRA.AttachImagesPlugin");

JIRA.AttachImagesPlugin.handler = (function() {

    // Initialization of HTML5 paste handler
    "use strict";
    var screenshotPasteHandler;

    AJS.EventQueue = AJS.EventQueue || [];

    screenshotPasteHandler = {
        REQUEST_TIMEOUT_MILLIS: 5 * 60 * 1000,

        screenshotFileUpload: {},
        screenshotFileUploadUri: null,

        dropHandlerConfigured: false,
        dropFeatureEnabled: true,

        $document: undefined,
        $window: undefined,
        $container: undefined,
        $fakeInput: undefined,

        uploadError: false,
        uploadErrorMsg: "",

        progressView: {
            hidden: false,
            progress: 0,
            old: 0,
            progressBarContainer: undefined,
            progressEl: undefined,
            container: undefined,
            options: undefined,
            staticProgress: undefined,

            initProgress: function() {
                this.container = this.buildContainer();
                this.progressEl = this.buildProgress();
                this.container.append(this.progressEl);

                this.options = { showPercentage: false, height: "2px" };

                this.progressBarContainer = AJS.$("#attach-screenshot-progress-container");
                this.progressBarContainer.empty();
                this.progressBarContainer.append(this.container);

                this.staticProgress = this.container;
                this.hidden = true;
            },

            finish: function () {
                this.value(100);
            },

            progressHandler: function(event) {
                var percentage = Math.round(event.loaded * 100 / event.total);
                screenshotPasteHandler.debug("progress = " + percentage);
                screenshotPasteHandler.progressView.value(percentage);
            },

            value: function(value) {
                if (value > 100) {
                    value = 100;
                } else if (value < 0) {
                    value = 0;
                }

                if (this.hidden) {
                    screenshotPasteHandler.debug("showing progressEl");
                    this.progressEl.show();
                    this.hidden = false;
                }

                if (this.old !== value) {
                    this.progressEl.progressBar(value, this.options);
                    if (value >= 100) {
                        this.progressEl.fadeOut();
                    }
                    this.old = value;
                }
            },

            buildContainer: function() {
                return AJS.$("<div>").addClass("file-progress");
            },

            buildProgress: function() {
                return AJS.$("<div>").attr("id", "attach-screenshot-upload-progress");
            }
        },

        dialogView: {
            pasteCatcher: {},
            presenter: undefined,

            getMaxSize : function() {
                return AJS.$("#attach-max-size").text();
            },

            getFileSize : function() {
                if (AJS.$.isPlainObject(this.presenter.screenshotFileUpload) && _.isEmpty(this.presenter.screenshotFileUpload)) {
                    return 0;
                }
                return this.presenter.screenshotFileUpload.size;
            },

            cleanGeneralErrors: function() {
                AJS.$("#error-attach-screenshot-image").closest(".field-group").remove();
            },

            cleanFileErrors: function() {
                AJS.$("#error-attachScreenshotFileName").remove();
            },

            displayErrors: function (errors) {
                // cleanup old error messages
                screenshotPasteHandler.dialogView.cleanFileErrors();
                screenshotPasteHandler.dialogView.cleanGeneralErrors();

                if ("compatibility" in errors) {
                    AJS.$("#attach-screenshot-inner-container").before(aui.message.error({
                        content: errors["compatibility"]
                    }));
                }
                if ("fileName" in errors) {
                    AJS.$("#attach-screenshot-filename-group").append("<div class=\"error\" id=\"error-attach-screenshot-filename\" data-field=\"attachscreenshotname\">" + errors["fileName"] + "</div>");
                }
                if ("fileUpload" in errors) {
                    AJS.$("#attach-screenshot-inner-container").append("<div class=\"field-group attach-screenshot-image-error\"><div class=\"error\" id=\"error-attach-screenshot-image\">" + errors["fileUpload"] + "</div></div>");
                }
            },

            appendBlobImage: function(blob) {
                // and use a URL or webkitURL (whichever is available to the browser)
                // to create a temporary URL to the object
                var URLObj = window.URL || window.webkitURL;
                var source = URLObj.createObjectURL(blob);
                this.presenter.screenshotFileUpload = blob;
                this.createImage(source);
            },

            generateFileName: function() {
                var currentTime = new Date().getTime();
                var tempFileName = "attach-screenshot-temp-" + currentTime;
            },

            /**
             * Creates and shows image
             * Supports:
             *  Chrome binary data from Webkit
             *  base64 encoded data from FireFox
             * @param uri image data
             */
            createImage: function (uri) {
                var pastedImage = new Image();
                pastedImage.onload = function () {
                    // You now have the image!
                };
                pastedImage.src = uri;

                this.presenter.screenshotToUpload = pastedImage;

                // Appending image to document
                var jqueryImage = AJS.$(pastedImage);
                jqueryImage.addClass("attach-screenshot-pasted-image");
                var screenshotContainer = AJS.$("#attach-screenshot-image-container");
                screenshotContainer.empty();
                screenshotContainer.append(jqueryImage);

                this.presenter.$fakeInput.focus();

                // handle async upload
                screenshotPasteHandler.imageCreatedHandler();
            },

            /**
             * Parse the input in the paste catcher element
             */
            checkInput: function () {
                var image,
                    node = screenshotPasteHandler.dialogView.pasteCatcher.childNodes[0];

                if (node) {
                    // If the user pastes an image, the src attribute
                    // will represent the image as a base64 encoded string.
                    if ("IMG" === node.tagName) {
                        image = node.src;
                    }

                    // Clear the inner html to make sure we're always getting the latest inserted content.
                    screenshotPasteHandler.dialogView.pasteCatcher.innerHTML = "";
                }

                if (!image) {
                    screenshotPasteHandler.$fakeInput.focus();
                }

                return image;
            },

            /**
             * onPaste handler, Either uses webkits clipboardData object on the paste event, or interprets
             * the data uri that has been embedded in the page by firefox.
             */
            onPaste: function (event) {
                this.presenter.debug("attach image - onPaste");
                screenshotPasteHandler.dialogView.cleanFileErrors();
                screenshotPasteHandler.dialogView.cleanGeneralErrors();

                // TODO: ensure target is not text fields - this prevents pasting text in text fields

                // We need to check if event.clipboardData is supported (Chrome)
                if (event && event.clipboardData && event.clipboardData.items) {
                    // Get the items from the clipboard
                    var items = event.clipboardData.items;

                    if (items) {
                        // Loop through all items, looking for any kind of image
                        for (var i = 0; i < items.length; i++) {
                            if (items[i].type.indexOf("image") !== -1) {

                                // We need to represent the image as a file,
                                this.appendBlobImage(items[i].getAsFile());
                            }
                        }
                    }
                    // If we can't handle clipboard data directly (Firefox),
                    // we need to read what was pasted from the contenteditable element
                } else {
                    setTimeout(function () {
                        var image = this.checkInput();

                        if (image) {
                            // Firefox image is base64 encoded - decoding while setting up the data
                            screenshotPasteHandler.screenshotFileUpload = screenshotPasteHandler.decodeBase64DataUri(image);
                            screenshotPasteHandler.dialogView.createImage(image);
                        }

                    }.bind(this), 0);
                }
            },

            getFakeInput: function() {
                return AJS.$("#attach-screenshot-fake-input");
            },

            getContainer: function() {
                return AJS.$("#attach-screenshot-image-container");
            },

            getIssueKey: function() {
                return AJS.$("input[name='id']").val();
            },

            getDocument: function() {
                return AJS.$(document);
            },

            getWindow: function() {
                return AJS.$(window);
            },

            getFileNameInput: function() {
                return AJS.$("#attachscreenshotname");
            },

            hasPngExtension: function (str) {
                var pattern=/\.png$/i; // REGEX: Ends with ".png". Case insensitive
                return pattern.test(str);
            },

            setFileToConvert: function (value) {
                AJS.$("input[name='filetoconvert']").val(value);
            },

            /**
             * Pasting into a content-editable element is the most cross-browser HTML5 approach.
             */
            buildPasteCatcher: function() {
                if (!document.getElementById("attach-screenshot-form")) return;
                var catcher = document.createElement("div");
                catcher.setAttribute("contenteditable", "true");
                catcher.style.width = 0;
                catcher.style.height = 0;
                document.getElementById("attach-screenshot-form").appendChild(catcher);
                return catcher;
            },

            _getFormSubmits: function() {
                return AJS.$("#attach-screenshot-form").find("button.aui-button");
            },

            disable: function() {
                this._getFormSubmits().attr("disabled", "disabled");
                return this;
            },
            enable: function() {
                this._getFormSubmits().removeAttr("disabled");
                return this;
            },
            isEnabled: function() {
                return !this._getFormSubmits().attr("disabled");
            },

            initDialog: function(presenter) {
                this.pasteCatcher = {};
                this.presenter = presenter;

                if (!JIRA.AttachImagesPlugin.polyfill.isRequired()) {
                    this.pasteCatcher = this.buildPasteCatcher();
                }
            }
        },

        initScreenshotPasteHandler: function () {
            var dialogView = screenshotPasteHandler.dialogView;

            screenshotPasteHandler.screenshotFileUpload = {};
            screenshotPasteHandler.resetUploadErrors();

            screenshotPasteHandler.dialogView.initDialog(screenshotPasteHandler.getPresenter());

            // Caching elements needed to fix tabbing
            screenshotPasteHandler.$document =  dialogView.getDocument();
            screenshotPasteHandler.$window = dialogView.getWindow();
            screenshotPasteHandler.$container = dialogView.getContainer();
            screenshotPasteHandler.$fakeInput = dialogView.getFakeInput();

            // Event Handlers
            screenshotPasteHandler.bindOnce(screenshotPasteHandler.$container, "click", screenshotPasteHandler.setFocusOnClickHandler);
            screenshotPasteHandler.bindOnce(screenshotPasteHandler.$fakeInput, "focus", screenshotPasteHandler.showFocusOnFieldHandler);
            screenshotPasteHandler.bindOnce(screenshotPasteHandler.$fakeInput, "blur", screenshotPasteHandler.hideFocusOnFieldHandler);
            screenshotPasteHandler.bindOnce(screenshotPasteHandler.$fakeInput, "keydown", screenshotPasteHandler.keyDownHandler);

            if (!JIRA.AttachImagesPlugin.browserIsSupported()) {
                dialogView.displayErrors({"compatibility": AJS.I18n.getText("attach.screenshot.browser.not.supported", "<a href=\"//www.google.com/chrome/browser/\">", "</a>", "<a href=\"//www.mozilla.org/firefox/\">", "</a>")});
            } else if (JIRA.AttachImagesPlugin.polyfill.isRequired()) {
                JIRA.AttachImagesPlugin.polyfill.install(document)
                    .done(function (executeAjaxUploadFromApplet) {
                        screenshotPasteHandler.bindOnce(screenshotPasteHandler.$document, "polyPaste", screenshotPasteHandler.polyPasteHandler);
                        if (JIRA.AttachImagesPlugin.polyfill.isRequiredForBinaryAjax()) {
                            screenshotPasteHandler.executeAjaxUpload = executeAjaxUploadFromApplet;
                        }
                    })
                    .fail(function (reason, message) {
                        dialogView.displayErrors({"compatibility": message});
                    });
            } else {
                screenshotPasteHandler.bindOnce(screenshotPasteHandler.$window, "paste", screenshotPasteHandler.pasteHandler);
            }
        },

        resetUploadErrors: function() {
            screenshotPasteHandler.uploadError = false;
            screenshotPasteHandler.uploadErrorMsg = undefined;
        },

        setUploadError: function(errorMsg) {
            screenshotPasteHandler.uploadError = true;
            screenshotPasteHandler.uploadErrorMsg = errorMsg;
        },

        /**
         * Ensures only one event is bound to the element
         * @param jqueryEl
         * @param eventName
         * @param handler
         */
        bindOnce: function(jqueryEl, eventName, handler) {
            jqueryEl.unbind(eventName, handler);
            jqueryEl.bind(eventName, handler);
        },

        showFocusOnFieldHandler: function() {
            screenshotPasteHandler.$container.addClass("focus");
        },

        hideFocusOnFieldHandler: function() {
            screenshotPasteHandler.$container.addClass("focus");
        },

        setFocusOnClickHandler: function() {
            screenshotPasteHandler.debug("attach image - focusingHandler");
            screenshotPasteHandler.$fakeInput.focus();
        },

        pasteHandler: function(event) {
            if(!screenshotPasteHandler.dialogView.isEnabled()) {
                screenshotPasteHandler.debug("attach image - onPaste not ready");
                return;
            }

            screenshotPasteHandler.debug("attach image - pasteHandler");
            screenshotPasteHandler.dialogView.onPaste(event.originalEvent);
        },

        polyPasteHandler: function(event, imageUri, imagePreviewUri) {
            if(!screenshotPasteHandler.dialogView.isEnabled()) {
                screenshotPasteHandler.debug("attach image - onPaste not ready");
                return;
            }

            screenshotPasteHandler.screenshotFileUpload = { length : 0 };
            screenshotPasteHandler.screenshotFileUploadUri = imageUri;
            screenshotPasteHandler.dialogView.createImage(imagePreviewUri);
        },

        keyDownHandler: function(event) {
            if (JIRA.AttachImagesPlugin.isPasteEvent(event)) {
                if (screenshotPasteHandler.dialogView.pasteCatcher.focus) {
                    screenshotPasteHandler.dialogView.pasteCatcher.focus();
                }
            }
        },

        getPresenter: function() {
            return screenshotPasteHandler;
        },

        imageCreatedHandler: function() {
            var tempFileName = screenshotPasteHandler.dialogView.generateFileName();
            screenshotPasteHandler.doAjaxUpload(this.dialogView.getIssueKey(), tempFileName);
        },

        getMimeType: function () {
            if (screenshotPasteHandler.screenshotFileUploadUri) {
                return "image/jpeg";
            }
            return "image/png";
        },

        createData: function() {
            if (JIRA.AttachImagesPlugin.polyfill.isRequired() && !JIRA.AttachImagesPlugin.polyfill.isRequiredForBinaryAjax()) {
                return this.decodeBase64DataUri(screenshotPasteHandler.screenshotFileUploadUri);
            }
            return screenshotPasteHandler.screenshotFileUpload;
        },

        /**
         * Logs message to the browser console
         * Do nothing if console object is not defined
         * @param msg string message
         */
        debug: function (msg) {
            if (!("undefined" === console)) {
                console.log(msg);
            }
        },

        hasCharacter: function(s, c) {
            return s.indexOf(c) != -1;
        },

        decodeBase64DataUri: function (uri) {
            var separator = ";base64,",
                prefix = uri.slice(0, uri.indexOf(separator)) + separator,
                base64encoded = uri.substring(prefix.length);
            return JIRA.AttachImagesPlugin.base64decode(base64encoded);
        },

        validateFileSize: function(errors) {
            var fileSize = screenshotPasteHandler.dialogView.getFileSize();
            var maxSize = screenshotPasteHandler.dialogView.getMaxSize();
            if(fileSize > maxSize) {
//                TODO: hack, using IninleAttach module, check if this a blessed way to go
                var sizes = AJS.InlineAttach.Text.fileSize(maxSize, fileSize);
                errors["fileUpload"] = AJS.I18n.getText("attach.screenshot.error.upload.too.big", sizes[1], sizes[0]);
            }
        },

        validateFormData: function(fileUpload, fileName) {
            var errors = {};

            if (AJS.$.isPlainObject(fileUpload) && _.isEmpty(fileUpload)) {
                errors["fileUpload"] = AJS.I18n.getText("attach.screenshot.specify.image");
            }
            screenshotPasteHandler.validateFileSize(errors);
            if (screenshotPasteHandler.uploadError) {
                errors["fileUpload"] = AJS.I18n.getText("attach.screenshot.cant.attach.upload.failed");
            }

            if ("" == fileName) {
                errors["fileName"] = AJS.I18n.getText("attach.screenshot.specify.file.name");
            }

            if (this.hasCharacter(fileName, '\\') ||
                this.hasCharacter(fileName, '/') ||
                this.hasCharacter(fileName, '\"') ||
                this.hasCharacter(fileName, ':') ||
                this.hasCharacter(fileName, '?') ||
                this.hasCharacter(fileName, '*') ||
                this.hasCharacter(fileName, '<') ||
                this.hasCharacter(fileName, '|') ||
                this.hasCharacter(fileName, '>'))  {

                errors["fileName"] = AJS.I18n.getText("attach.screenshot.error.illegal.file.name");
            }
            return errors;
        },

        doAjaxUpload : function(issueKey, fileName) {
            this.debug("attach image - doAjaxUpload");

            screenshotPasteHandler.dialogView.disable();

            var errors = [];
            screenshotPasteHandler.validateFileSize(errors);
            if (!_.isEmpty(errors)) {
                screenshotPasteHandler.dialogView.displayErrors(errors);
                screenshotPasteHandler.dialogView.enable();

                // Prevent form from uploading
                return;
            }

            var data = screenshotPasteHandler.createData(),
                mimeType = screenshotPasteHandler.getMimeType(),
                secureToken = AJS.$("#attach-screenshot-form").data("attach-secure-token");

            var requestUrl = AJS.contextPath() + "/rest/internal/1.0/AttachTemporaryFile"+
                (JIRA.AttachImagesPlugin.polyfill.isRequired() ? "/secure?" : "?") +
                "filename=" + encodeURIComponent(fileName) + "&" +
                "atl_token=" + encodeURIComponent(atl_token()) + "&" +
                "issueId=" + encodeURIComponent(this.dialogView.getIssueKey()) +
                (secureToken ? ("&secureToken=" + encodeURIComponent(secureToken)) : "");

            screenshotPasteHandler.resetUploadErrors();
            screenshotPasteHandler.executeAjaxUpload(data, requestUrl, mimeType).progress(function(val) {
                if(val == 'init') {
                    screenshotPasteHandler.progressView.initProgress()
                }
            }).done(function(val) {
                if(typeof val == "string") {
                    val = JSON.parse(val);
                }

                screenshotPasteHandler.dialogView.setFileToConvert(val.id);
            }).fail(function(jqXHR, textStatus, msg, smartAjaxResult) {
                var errMsg;
                if (smartAjaxResult.statusText == "abort") {
                    errMsg = AJS.I18n.getText("attach.screenshot.request.aborted");
                } else {
                    errMsg = JIRA.SmartAjax.buildSimpleErrorContent(smartAjaxResult, { alert : false });
                }
                screenshotPasteHandler.setUploadError(errMsg);
                screenshotPasteHandler.dialogView.displayErrors({"fileUpload" : errMsg});
            }).always(function() {
                var deferred = AJS.$.Deferred();

                // polyfill is required so we need a new secureToken
                if(JIRA.AttachImagesPlugin.polyfill.isRequired()) {
                    screenshotPasteHandler.reloadSecureToken(deferred);
                } else {
                    deferred.resolve();
                }

                deferred.then(function() {
                    screenshotPasteHandler.dialogView.enable();
                    // Progress Listener cleanup
                    screenshotPasteHandler.progressView.finish();
                });
            }).progress(function(val) {
                if(val != 'init') {
                    screenshotPasteHandler.progressView.progressHandler(val);
                }
            });
        },

        executeAjaxUpload: function (data, requestUrl, mimeType) {
            var deferred = AJS.$.Deferred();

            JIRA.SmartAjax.makeRequest({
                type: 'POST',
                url: requestUrl,
                contentType: mimeType,
                processData: false,
                data: data,
                timeout : screenshotPasteHandler.REQUEST_TIMEOUT_MILLIS,
                success: deferred.resolve.bind(deferred),
                error: deferred.reject.bind(deferred),
                xhr: function() {
                    var xhr = AJS.$.ajaxSettings.xhr();

                    deferred.notify('init');
                    xhr.upload.addEventListener("progress", deferred.notify.bind(deferred));
                    return xhr;
                }
            });

            deferred.always(function() {
                AJS.$.ajaxSettings.xhr().removeEventListener("progress", screenshotPasteHandler.progressView.progressHandler);
            });

            return deferred;
        },

        reloadSecureToken: function(deferred) {
            var formUrl = AJS.contextPath() + "/secure/ShowAttachScreenshotFormAction!default.jspa?"+
                "id=" + encodeURIComponent(this.dialogView.getIssueKey()) +
                "&inline=true&decorator=dialog&_="+(new Date().getTime());

            AJS.$.get(formUrl).then(function(response) {
                var newToken = AJS.$(response).find("#attach-screenshot-form").data("attach-secure-token");
                if(newToken) {
                    AJS.$("#attach-screenshot-form").data("attach-secure-token", newToken);
                    deferred.resolve(newToken);
                } else {
                    deferred.reject();
                }
            }, _.bind(deferred.reject, deferred));
        },

        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        handleDrop: function (event) {
            screenshotPasteHandler.debug("handleDrop");


            screenshotPasteHandler.dialogView.disable();

            var errors = [];
            screenshotPasteHandler.validateFileSize(errors);
            if (!_.isEmpty(errors)) {
                screenshotPasteHandler.dialogView.displayErrors(errors);
                screenshotPasteHandler.dialogView.enable();

                // Prevent form from uploading
                return;
            }


            var data = screenshotPasteHandler.createData();

            var requestUrl = AJS.contextPath() + "/rest/internal/1.0/AttachTemporaryFile/secure?" +
                "filename=" + fileName + "&" +
                "atl_token=" + atl_token() + "&" +
                "issueId=" + this.dialogView.getIssueKey() + "&" +
                "secureToken=" + AJS.$("#attach-screenshot-form").data("attach-secure-token");

            screenshotPasteHandler.resetUploadErrors();
            /* For each dropped file. */
            var files = this.files || event.dataTransfer.files;

            if (files === "undefined" || files.length == 0) {
                alert(AJS.I18n.getText("dnd.attachment.no.files"));
            } else {
                screenshotPasteHandler.uploadFiles(files);
            }
        },

        uploadFiles: function(files) {
            screenshotPasteHandler.debug("uploadFiles");

            var f = {};
            for (var idx = 0; idx < files.length; ++idx) {
                f = files[idx];
                break;
            }
            this.dialogView.appendBlobImage(f);
        }
    };


    // Upload and Cancel Button Handlers
    AJS.$(document).ready(function(){
        "use strict";

        var AttachImageDialog = JIRA.FormDialog.extend({
            options: {}
        });

        var delayShowUntil = AJS.$.Deferred();

        if (!JIRA.AttachImagesPlugin.browserIsSupported() ||
            !JIRA.AttachImagesPlugin.polyfill.isRequired()) {
            delayShowUntil.resolve();
        }

        // Shows the Attach Screenshot in a Popup
        JIRA.Dialogs.attachScreenshotDialog = new AttachImageDialog({
            id: "attach-screenshot-dialog",
            trigger: "a.issueaction-attach-image",
            isIssueDialog: true,
            onContentRefresh : function attachScreenshotContentRefresh() {

                this.$form.bind("before-submit", function attachScreenshotContentRefreshBeforeSubmit(e) {
                    var errors = screenshotPasteHandler.validateFormData(screenshotPasteHandler.screenshotFileUpload, AJS.$.trim(screenshotPasteHandler.dialogView.getFileNameInput().val()));
                    if (screenshotPasteHandler.dialogView.getFileSize() == 0) {
                        e.preventDefault();
                        return false;
                    } else if (!_.isEmpty(errors)) {
                        screenshotPasteHandler.dialogView.displayErrors(errors);
                        e.preventDefault();
                        return false;
                    }
                    return true;
                });

            },
            delayShowUntil: function() {
                if (JIRA.AttachImagesPlugin.polyfill.isRequired()) {
                    JIRA.AttachImagesPlugin.polyfill.install(document)
                        .done(function() {
                            AttachImageDialog.prototype._submitForm = JIRA.AttachImagesPlugin.polyfill.proxyAjaxRequest(AttachImageDialog.prototype._submitForm, function() {
                                return AJS.$("#attach-screenshot-form").data("submit-secure-token");
                            });
                        })
                        .always(function() {
                            delayShowUntil.resolve();
                        });
                }
                return delayShowUntil;
            }
        });

        AJS.$(document).bind("dialogContentReady", function (event, dialog) {
            // Ensure that the dialog is *ours*, and that it's showing the Attach screen shot form. It's possible that
            // it's populated with an error from the server, in which case we wouldn't want to "init" it.
            if (dialog === JIRA.Dialogs.attachScreenshotDialog
                && document.getElementById("attach-screenshot-form") !== null) {
                screenshotPasteHandler.initScreenshotPasteHandler();
            }
        });

        AJS.$(document).ready(function() {
            // Atlassian Analytics - Capture click events
            AJS.$(document).on("click", "#attach-screenshot-html5", function () {
                AJS.EventQueue.push({
                    name: "attach.screenshot.html5.display",
                    properties: {}
                });
            });
        });

        if (screenshotPasteHandler.dropFeatureEnabled) {
//        TODO: check does not break drag-n-drop plugin attachments
//        TODO: check dndAttachment.configured is true for drag-n-drop plugin

            if (!screenshotPasteHandler.dropHandlerConfigured) {
                AJS.$(document)
                    .on("dragover", function () {
                        //screenshotPasteHandler.debug("document.dragover");
                    })
                    .on("dragleave drop", function () {
                        //screenshotPasteHandler.debug("document.dragleave drop");
                    })
                    .on("dragover drop", false);

                AJS.$(document).on("drop", ".attach-screenshot-image-container", function (e) {
                    //screenshotPasteHandler.debug("document.drop .attach-screenshot-image-container");
                    screenshotPasteHandler.handleDrop(e);
                    screenshotPasteHandler.dropHandlerConfigured = true;
                });

            }
        }

    });

    return screenshotPasteHandler;
})();


