package com.atlassian.plugins.whitelist.ui;

import com.atlassian.plugin.webresource.WebResourceManager;
import com.atlassian.plugins.whitelist.WhitelistService;
import com.atlassian.sal.api.auth.LoginUriProvider;
import com.atlassian.sal.api.user.UserKey;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.api.websudo.WebSudoManager;
import com.atlassian.sal.api.websudo.WebSudoSessionException;
import com.atlassian.soy.renderer.SoyException;
import com.atlassian.soy.renderer.SoyTemplateRenderer;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang.CharEncoding;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.core.CacheControl;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import java.io.IOException;
import java.net.URI;

/**
 * @since 1.0
 */
public class WhitelistServlet extends HttpServlet
{
    private static final String WHITELIST_TEMPLATE_KEY = "com.atlassian.plugins.atlassian-whitelist-ui-plugin:whitelist-bootstrap-template";
    private static final String WHITELIST_WEB_RESOURCES_KEY = "com.atlassian.plugins.atlassian-whitelist-ui-plugin:whitelist-web-resources";
    private static final String WHITELIST_SOY_TEMPLATE = "com.atlassian.plugins.whitelist.ui.whitelistPage";

    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final Logger LOGGER = LoggerFactory.getLogger(WhitelistServlet.class);

    private final SoyTemplateRenderer soyTemplateRenderer;
    private final WebResourceManager webResourceManager;
    private final WhitelistService whitelistService;
    private final WebSudoManager webSudoManager;
    private final UserManager userManager;
    private final LoginUriProvider loginUriProvider;
    private final WhitelistBeanService whitelistBeanService;

    public WhitelistServlet(final SoyTemplateRenderer soyTemplateRenderer, final WebResourceManager webResourceManager,
                            final WhitelistService whitelistService, final WebSudoManager webSudoManager,
                            final UserManager userManager, final LoginUriProvider loginUriProvider,
                            final WhitelistBeanService whitelistBeanService)
    {
        this.soyTemplateRenderer = soyTemplateRenderer;
        this.webResourceManager = webResourceManager;
        this.whitelistService = whitelistService;
        this.webSudoManager = webSudoManager;
        this.userManager = userManager;
        this.loginUriProvider = loginUriProvider;
        this.whitelistBeanService = whitelistBeanService;
    }

    @Override
    public void doGet(final HttpServletRequest req, final HttpServletResponse resp) throws IOException
    {
        try
        {
            webSudoManager.willExecuteWebSudoRequest(req);

            if (isCurrentUserSysAdmin(req))
            {
                renderResponse(req, resp);
            }
            else
            {
                resp.sendRedirect(loginUriProvider.getLoginUri(URI.create(req.getRequestURL().toString())).toASCIIString());
            }
        }
        catch (WebSudoSessionException wse)
        {
            webSudoManager.enforceWebSudoProtection(req, resp);
        }
    }

    private boolean isCurrentUserSysAdmin(final HttpServletRequest req)
    {
        final UserKey remoteUserKey = userManager.getRemoteUserKey(req);
        return remoteUserKey != null && userManager.isSystemAdmin(remoteUserKey);
    }

    private void renderResponse(final HttpServletRequest req, final HttpServletResponse resp) throws IOException
    {
        resp.setContentType(MediaType.TEXT_HTML);
        resp.setCharacterEncoding(CharEncoding.UTF_8);
        resp.setHeader(HttpHeaders.CACHE_CONTROL, noCache());
        webResourceManager.requireResource(WHITELIST_WEB_RESOURCES_KEY);

        try
        {
            final Iterable<WhitelistBean> beans = whitelistBeanService.getAll();
            final ImmutableMap<String, Object> params = ImmutableMap.<String, Object>of(
                    "contextPath", req.getContextPath(),
                    "enabled", whitelistService.isWhitelistEnabled(),
                    "data", OBJECT_MAPPER.writeValueAsString(beans));
            soyTemplateRenderer.render(resp.getWriter(), WHITELIST_TEMPLATE_KEY, WHITELIST_SOY_TEMPLATE, params);
        }
        catch (SoyException e)
        {
            LOGGER.info("Failed to render soy template '{}': {}", WHITELIST_SOY_TEMPLATE, e.getMessage());
            LOGGER.debug("Failed to render soy template '" + WHITELIST_SOY_TEMPLATE + "' contained in resource '" + WHITELIST_TEMPLATE_KEY + "'", e);
            resp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, "Failed to render response");
        }
    }

    private static String noCache()
    {
        final CacheControl cacheControl = new CacheControl();
        cacheControl.setNoCache(true);
        cacheControl.setNoStore(true);
        return cacheControl.toString();
    }

}
