package com.atlassian.plugins.whitelist.ui;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.ApplicationLinkService;
import com.atlassian.applinks.api.TypeNotInstalledException;
import com.atlassian.applinks.spi.application.TypeId;
import com.atlassian.plugins.whitelist.WhitelistRule;
import com.atlassian.plugins.whitelist.WhitelistType;
import com.google.common.collect.ComparisonChain;

import javax.annotation.Nullable;
import java.util.Comparator;

/**
 * @since 1.1
 */
public class WhitelistRuleComparator implements Comparator<WhitelistRule>
{
    private final ApplicationLinkService applicationLinkService;

    public WhitelistRuleComparator(final ApplicationLinkService applicationLinkService)
    {
        this.applicationLinkService = applicationLinkService;
    }

    @Override
    public int compare(final WhitelistRule o1, final WhitelistRule o2)
    {
        if (o1.getType() == WhitelistType.APPLICATION_LINK && o2.getType() == WhitelistType.APPLICATION_LINK)
        {
            return compareApplicationLinks(o1, o2);
        }
        else
        {
            return ComparisonChain.start()
                    .compare(o1.getType(), o2.getType())
                    .compare(o1.getExpression(), o2.getExpression(), String.CASE_INSENSITIVE_ORDER)
                    .result();
        }
    }

    private int compareApplicationLinks(final WhitelistRule o1, final WhitelistRule o2)
    {
        final ApplicationLink applicationLink1 = getApplicationName(o1);
        final ApplicationLink applicationLink2 = getApplicationName(o2);
        if (applicationLink1 == null || applicationLink2 == null)
        {
            return o1.getExpression().compareToIgnoreCase(o2.getExpression());
        }

        return ComparisonChain.start()
                .compare(TypeId.getTypeId(applicationLink1.getType()), TypeId.getTypeId(applicationLink2.getType()))
                .compare(applicationLink1.getName(), applicationLink2.getName())
                .result();
    }

    @Nullable
    private ApplicationLink getApplicationName(final WhitelistRule whitelistRule)
    {
        try
        {
            final ApplicationId applicationId = new ApplicationId(whitelistRule.getExpression());
            return applicationLinkService.getApplicationLink(applicationId);
        }
        catch (TypeNotInstalledException e)
        {
            return null;
        }
    }

}