package com.atlassian.plugins.whitelist.ui;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.ApplicationLinkService;
import com.atlassian.applinks.api.TypeNotInstalledException;
import com.atlassian.plugins.whitelist.WhitelistRule;
import com.atlassian.plugins.whitelist.WhitelistService;
import com.atlassian.plugins.whitelist.WhitelistType;
import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Ordering;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Predicates.notNull;

/**
 * @since 1.0
 */
class WhitelistBeanServiceImpl implements WhitelistBeanService
{
    private static final Logger logger = LoggerFactory.getLogger(WhitelistBeanServiceImpl.class);
    private final ApplicationLinkService applicationLinkService;
    private final WhitelistService whitelistService;
    private WhitelistRuleComparator whitelistRuleComparator;
    private final ImmutableMap<WhitelistType, Function<WhitelistRule, WhitelistBean>> mappings = ImmutableMap.of(
            WhitelistType.APPLICATION_LINK, createApplicationLinkRule(),
            WhitelistType.EXACT_URL, createUrlExpressionRule(),
            WhitelistType.WILDCARD_EXPRESSION, createUrlExpressionRule(),
            WhitelistType.REGULAR_EXPRESSION, createUrlExpressionRule(),
            WhitelistType.DOMAIN_NAME, createUrlExpressionRule()
    );

    public WhitelistBeanServiceImpl(final ApplicationLinkService applicationLinkService, final WhitelistService whitelistService, final WhitelistRuleComparator whitelistRuleComparator)
    {
        this.applicationLinkService = applicationLinkService;
        this.whitelistService = whitelistService;
        this.whitelistRuleComparator = whitelistRuleComparator;
    }

    @Override
    public WhitelistBean add(final WhitelistBean whitelistBean)
    {
        return asBean(whitelistService.add(whitelistBean.asRule()));
    }

    @Override
    public WhitelistBean update(final int id, final WhitelistBean whitelistBean)
    {
        final WhitelistRule existingWhitelistRule = whitelistService.get(id);
        checkNotNull(existingWhitelistRule, "Whitlist rule with id '" + id + "' not existing.");
        final WhitelistRule updatedRule = whitelistBean.populateWith(existingWhitelistRule);
        final WhitelistRule whitelistRule = whitelistService.update(updatedRule);
        return asBean(whitelistRule);
    }

    @Override
    public List<WhitelistBean> getAll()
    {
        return asBeans(Ordering.from(whitelistRuleComparator).immutableSortedCopy(whitelistService.getAll()));
    }

    private List<WhitelistBean> asBeans(final List<WhitelistRule> whitelistRules)
    {
        return ImmutableList.copyOf(Collections2.filter(Lists.transform(whitelistRules, asBean()), notNull()));
    }

    private WhitelistBean asBean(final WhitelistRule whitelistRule)
    {
        checkNotNull(whitelistRule, "whitelistRule");
        final WhitelistType type = whitelistRule.getType();
        final Function<WhitelistRule, WhitelistBean> whitelistRuleDataWhitelistBeanFunction = mappings.get(type);
        if (whitelistRuleDataWhitelistBeanFunction == null)
        {
            logger.warn("No mapping found for whitelist type '{}', ignoring data '{}'.", type, whitelistRule);
            return null;
        }
        else
        {
            return whitelistRuleDataWhitelistBeanFunction.apply(whitelistRule);
        }
    }

    private Function<WhitelistRule, WhitelistBean> asBean()
    {
        return new Function<WhitelistRule, WhitelistBean>()
        {
            @Override
            public WhitelistBean apply(@Nullable final WhitelistRule input)
            {
                if (input == null)
                {
                    return null;
                }

                return WhitelistBeanServiceImpl.this.asBean(input);
            }
        };
    }

    private Function<WhitelistRule, WhitelistBean> createApplicationLinkRule()
    {
        return new Function<WhitelistRule, WhitelistBean>()
        {
            @Override
            public WhitelistBean apply(final WhitelistRule input)
            {
                final String applicationId = input.getExpression();
                try
                {
                    final ApplicationLink applicationLink = applicationLinkService.getApplicationLink(new ApplicationId(applicationId));
                    if (applicationLink == null)
                    {
                        logger.warn("Failed to resolved application link with application id '" + applicationId + "'; maybe it has been removed and the whitelist was not updated?");
                        return null;
                    }
                    else
                    {
                        final String applicationLinkName = applicationLink.getName();
                        final URI displayUrl = applicationLink.getDisplayUrl();
                        final URI iconUrl = applicationLink.getType().getIconUrl();
                        return WhitelistBean.builder()
                                .id(input.getId())
                                .expression(String.format("%s (%s)", applicationLinkName, displayUrl.toString()))
                                .type(WhitelistType.APPLICATION_LINK)
                                .iconUrl(iconUrl != null ? iconUrl.toString() : "")
                                .allowInbound(input.isAllowInbound())
                                .build();
                    }
                }
                catch (TypeNotInstalledException e)
                {
                    logger.warn("Failed to look up application link with application id '" + applicationId + "': the specified application link's type is not currently installed.");
                    return null;
                }
            }
        };
    }

    private Function<WhitelistRule, WhitelistBean> createUrlExpressionRule()
    {
        return new Function<WhitelistRule, WhitelistBean>()
        {
            @Override
            public WhitelistBean apply(final WhitelistRule input)
            {
                return WhitelistBean.builder().from(input).iconUrl(generateIconUrl(input, "")).build();
            }
        };
    }

    private String generateIconUrl(final WhitelistRule whitelistRule, final String defaultIconUrl)
    {
        if (whitelistRule.getType() == WhitelistType.EXACT_URL || whitelistRule.getType() == WhitelistType.DOMAIN_NAME)
        {
            return generateFavIconUrl(whitelistRule.getExpression(), defaultIconUrl);
        }
        else
        {
            return defaultIconUrl;
        }
    }

    private String generateFavIconUrl(String expression, String defaultIconUrl)
    {
        try
        {
            final URI uri = new URI(expression);
            return uri.getScheme() + "://" + uri.getAuthority() + "/favicon.ico";
        }
        catch (URISyntaxException e)
        {
            return defaultIconUrl;
        }
    }
}
