package com.atlassian.plugins.whitelist.applinks;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.event.ApplicationLinkAddedEvent;
import com.atlassian.applinks.api.event.ApplicationLinkDeletedEvent;
import com.atlassian.applinks.api.event.ApplicationLinkDetailsChangedEvent;
import com.atlassian.applinks.api.event.ApplicationLinksIDChangedEvent;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.event.events.PluginFrameworkStartedEvent;
import com.atlassian.plugins.whitelist.ImmutableWhitelistRule;
import com.atlassian.plugins.whitelist.WhitelistManager;
import com.atlassian.plugins.whitelist.WhitelistRule;
import com.atlassian.plugins.whitelist.WhitelistType;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import java.util.Collection;
import java.util.LinkedList;

import static com.atlassian.plugins.whitelist.WhitelistRulePredicates.withExpression;
import static com.atlassian.plugins.whitelist.WhitelistRulePredicates.withType;
import static com.atlassian.plugins.whitelist.applinks.ApplicationLinkWhitelistRule.getExpressionFrom;
import static com.google.common.base.Predicates.and;

/**
 * Keep the existing application links in sync with the stored ones.
 * @since 1.0
 */
public class ApplicationLinkChangeListener implements InitializingBean, DisposableBean
{
    private static final Logger logger = LoggerFactory.getLogger(ApplicationLinkChangeListener.class);

    private final EventPublisher eventPublisher;
    private final WhitelistManager whitelistManager;
    private final TransactionTemplate transactionTemplate;
    private final CachingApplicationLinkService cachingApplicationLinkService;

    private boolean started;

    public ApplicationLinkChangeListener(final EventPublisher eventPublisher, final WhitelistManager whitelistManager, final TransactionTemplate transactionTemplate, final CachingApplicationLinkService cachingApplicationLinkService)
    {
        this.eventPublisher = eventPublisher;
        this.whitelistManager = whitelistManager;
        this.transactionTemplate = transactionTemplate;
        this.cachingApplicationLinkService = cachingApplicationLinkService;
    }

    @EventListener
    @SuppressWarnings("UnusedDeclaration")
    public void onApplicationLinkAdded(final ApplicationLinkAddedEvent event)
    {
        if (started)
        {
            add(event.getApplicationLink());
            cachingApplicationLinkService.clear();
        }
    }

    @EventListener
    @SuppressWarnings("UnusedDeclaration")
    public void onApplicationLinkDetailsChangedEvent(final ApplicationLinkDetailsChangedEvent event)
    {
        if (started)
        {
            cachingApplicationLinkService.clear();
        }
    }

    @EventListener
    @SuppressWarnings("UnusedDeclaration")
    public void onApplicationLinkIdChanged(final ApplicationLinksIDChangedEvent event)
    {
        if (started)
        {
            update(event.getOldApplicationId(), event.getApplicationLink());
            cachingApplicationLinkService.clear();
        }
    }

    @EventListener
    @SuppressWarnings("UnusedDeclaration")
    public void onApplicationLinkDeleted(final ApplicationLinkDeletedEvent event)
    {
        if (started)
        {
            remove(event.getApplicationLink().getId());
            cachingApplicationLinkService.clear();
        }
    }

    @EventListener
    @SuppressWarnings("UnusedDeclaration")
    public void onPluginFrameworkStarted(final PluginFrameworkStartedEvent event)
    {
        final Collection<String> whitelistedApplicationLinkIds = Collections2.transform(Collections2.filter(whitelistManager.getAll(), withType(WhitelistType.APPLICATION_LINK)), new Function<WhitelistRule, String>()
        {
            @Override
            public String apply(final WhitelistRule whitelistRule)
            {
                return whitelistRule.getExpression();
            }
        });
        final Iterable<ApplicationLink> nonWhitelistedApplicationLinks = Iterables.filter(cachingApplicationLinkService.getApplicationLinks(), new Predicate<ApplicationLink>()
        {
            @Override
            public boolean apply(final ApplicationLink applicationLink)
            {
                return !whitelistedApplicationLinkIds.contains(applicationLink.getId().get());
            }
        });
        add(nonWhitelistedApplicationLinks);

        started = true;
    }

    @Override
    public void afterPropertiesSet()
    {
        eventPublisher.register(this);
    }

    @Override
    public void destroy()
    {
        eventPublisher.unregister(this);
    }

    private void add(final ApplicationLink applicationLink)
    {
        transactionTemplate.execute(new TransactionCallback<Void>()
        {
            @Override
            public Void doInTransaction()
            {
                // Only add a whitelist rule for applinks that are not already being whitelisted, else we will end up
                // with duplicates (which isn't really a big problem)!
                // This can occur in rare circumstances where this plugin is disabled and then later re-enabled (even
                // though this is a system plugin and should never be disabled in the first place!).
                final Collection<WhitelistRule> existingRules = findAllWhitelistedApplicationLinksWithId(applicationLink.getId());
                if (existingRules.isEmpty())
                {
                    whitelistManager.add(new ApplicationLinkWhitelistRule(applicationLink));
                }
                return null;
            }
        });
    }

    private void add(final Iterable<ApplicationLink> applicationLinks)
    {
        transactionTemplate.execute(new TransactionCallback<Void>()
        {
            @Override
            public Void doInTransaction()
            {
                for (ApplicationLink applicationLink : applicationLinks)
                {
                    whitelistManager.add(new ApplicationLinkWhitelistRule(applicationLink));
                }
                return null;
            }
        });
    }

    private void update(final ApplicationId oldApplicationId, final ApplicationLink applicationLink)
    {
        transactionTemplate.execute(new TransactionCallback<Void>()
        {
            @Override
            public Void doInTransaction()
            {
                final LinkedList<WhitelistRule> applinksWithOldApplicationId = Lists.newLinkedList(findAllWhitelistedApplicationLinksWithId(oldApplicationId));
                final WhitelistRule first = applinksWithOldApplicationId.pollFirst();
                if (first != null)
                {
                    final WhitelistRule expression = ImmutableWhitelistRule.builder().copyOf(first).expression(getExpressionFrom(applicationLink)).build();
                    whitelistManager.update(expression);
                }
                whitelistManager.removeAll(applinksWithOldApplicationId);
                return null;
            }
        });
    }

    private void remove(final ApplicationId applicationId)
    {
        transactionTemplate.execute(new TransactionCallback<Void>()
        {
            @Override
            public Void doInTransaction()
            {
                final Collection<WhitelistRule> applinksWithId = findAllWhitelistedApplicationLinksWithId(applicationId);
                whitelistManager.removeAll(applinksWithId);
                return null;
            }
        });
    }

    /**
     * Make sure this method is wrapped in a transaction.
     * @param applicationId the application id of the requested entry
     * @return all entries that have the type #{@link WhitelistType#APPLICATION_LINK} and the given applicationId as
     * expression.
     */
    private Collection<WhitelistRule> findAllWhitelistedApplicationLinksWithId(final ApplicationId applicationId)
    {
        final String expression = getExpressionFrom(applicationId);
        final WhitelistType type = WhitelistType.APPLICATION_LINK;
        final Collection<WhitelistRule> result = Collections2.filter(whitelistManager.getAll(), and(withType(type), withExpression(expression)));
        if (result.size() > 1)
        {
            logger.warn("Found more than one whitelist entry for the application link with id '{}'.", applicationId);
        }
        return result;
    }

}
