package com.atlassian.webresource.plugin.rest;

import com.atlassian.plugin.webresource.PluginResourceLocator;
import com.atlassian.plugin.webresource.WebResourceIntegration;
import com.atlassian.plugin.webresource.impl.Globals;
import com.atlassian.plugin.webresource.impl.config.Config;
import com.atlassian.plugin.webresource.impl.snapshot.RootPage;
import com.atlassian.plugin.webresource.prebake.PrebakeWebResourceAssemblerFactory;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;
import com.atlassian.webresource.api.assembler.resource.PrebakeError;
import com.atlassian.webresource.plugin.prebake.discovery.PreBakeState;
import com.atlassian.webresource.plugin.prebake.discovery.RootPageCrawler;
import com.atlassian.webresource.plugin.prebake.discovery.SuperBatchCrawler;
import com.atlassian.webresource.plugin.prebake.discovery.TaintedResource;
import com.atlassian.webresource.plugin.prebake.discovery.WebResourcePreBaker;
import com.atlassian.webresource.plugin.prebake.resources.HttpResourceCollector;
import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;
import java.io.File;
import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static com.atlassian.webresource.plugin.prebake.util.PreBakeUtil.BUNDLE_ZIP_FILE;
import static com.atlassian.webresource.plugin.prebake.util.PreBakeUtil.BUNDLE_ZIP_URI_PATH;
import static javax.ws.rs.core.Response.Status;
import static javax.ws.rs.core.Response.ok;
import static javax.ws.rs.core.Response.status;

/**
 * <p>
 * REST end-point responsible for web-resource pre-bake operation handling.
 * </p>
 * All resources will only be available when a system property named according to {@link Config#PREBAKE_FEATURE_ENABLED}
 * is set "true", otherwise {@link Status#FORBIDDEN} status will be returned.
 *
 * @since v3.5.0
 */
@AnonymousAllowed
@Path("prebake")
public final class PreBakeResource {

    private static final Logger log = LoggerFactory.getLogger(PreBakeResource.class);

    private final WebResourcePreBaker preBaker;
    private final Config config;
    private final Globals globals;

    @VisibleForTesting
    @Context
    UriInfo info;

    @SuppressWarnings("deprecation")
    public PreBakeResource(
            final WebResourceIntegration webResourceIntegration,
            final PrebakeWebResourceAssemblerFactory webResourceAssemblerFactory,
            final PluginResourceLocator pluginResourceLocator) {
        this(
                pluginResourceLocator.temporaryWayToGetGlobalsDoNotUseIt().getConfig(),
                pluginResourceLocator.temporaryWayToGetGlobalsDoNotUseIt(),
                webResourceIntegration,
                webResourceAssemblerFactory
        );
    }

    @VisibleForTesting
    PreBakeResource(
            final Config config,
            final Globals globals,
            final WebResourceIntegration webResourceIntegration,
            final PrebakeWebResourceAssemblerFactory webResourceAssemblerFactory) {
        this.globals = globals;
        this.config = config;
        this.preBaker = new WebResourcePreBaker(
                config,
                webResourceIntegration,
                webResourceAssemblerFactory,
                new HttpResourceCollector()
        );
    }

    /**
     * Initiates web-resources pre-baking operation.
     *
     * <p>In order to maintain backwards compatibility with the previous behaviour, this rest endpoint is hardwired to
     * prebake only the superbatch. It will be removed, or merged with the {@code prebakeRootPages} endpoint, once
     * there's at least one root-page declared in the JIRA XML descriptors (see APDEX-895 and APDEX-922)
     *
     * @param request Command for pre-baker. Not used at the moment
     * @return Outcome summary of the prebake operation
     */
    @Path("/state")
    @PUT
    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    public final Response put(PreBakeResult request) {
        return whenPreBakeIsEnabled(() -> {
            preBaker.start(new SuperBatchCrawler(preBaker.getWebResourceAssemblerFactory()));
            return ok(new PreBakeResult(
                    PreBakeState.RUNNING,
                    "" + URI.create(info.getBaseUri() + BUNDLE_ZIP_URI_PATH).normalize(),
                    preBaker.getBatch().getVersion(),
                    preBaker.getBatch().getTaintedResources())).build();
        });
    }

    /**
     * REST endpoint for prebaking resoures declared as root-page dependencies
     *
     * @param request Command for the prebaker. Not used at the moment.
     * @return Outcome summary of the prebake operation
     */
    @Path("/start")
    @PUT
    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    public final Response prebakeRootPages(PreBakeResult request) {
        return whenPreBakeIsEnabled(() -> {
            Iterable<RootPage> rootPages = globals.getSnapshot().getAllRootPages();
            preBaker.start(new RootPageCrawler(preBaker.getWebResourceAssemblerFactory(), rootPages));
            return ok(new PreBakeResult(
                    PreBakeState.RUNNING,
                    "" + URI.create(info.getBaseUri() + BUNDLE_ZIP_URI_PATH).normalize(),
                    preBaker.getBatch().getVersion(),
                    preBaker.getBatch().getTaintedResources())).build();
        });
    }

    /**
     * Shows current state ({@link PreBakeState}) of pre-baking process.
     *
     * @return result of pre-bake operation.
     */
    @Path("/state")
    @GET
    @Produces({MediaType.APPLICATION_JSON})
    public final Response get() {
        return whenPreBakeIsEnabled(() ->
                ok(new PreBakeResult(
                        preBaker.getState(),
                        "" + URI.create(info.getBaseUri() + BUNDLE_ZIP_URI_PATH).normalize(),
                        preBaker.getBatch().getVersion(),
                        preBaker.getBatch().getTaintedResources())).build());
    }

    /**
     * Provides bundle.zip with product state, pre-baked web-resources and mappings.
     *
     * @return zipped bundle.
     */
    @Path("/" + BUNDLE_ZIP_FILE)
    @GET
    @Produces({MediaType.APPLICATION_OCTET_STREAM})
    public final Response bundle() {
        return whenPreBakeIsEnabled(() -> {
            PreBakeState state = preBaker.getState();
            if (PreBakeState.DONE == state) {
                return
                        ok(new File(preBaker.getBundleDir(), preBaker.getBatch().getBundle()))
                                .header("Content-Disposition", "attachment; filename=" + BUNDLE_ZIP_FILE)
                                .build();
            } else {
                log.debug("Bundle is not ready, current state is [{}]", state);
                return status(Status.NOT_FOUND).build();
            }
        });
    }

    public static class PreBakeResult {
        public PreBakeState state;
        public String bundle;
        public String productStateHash;
        public List<TaintedResourceREST> taintedResources;

        public PreBakeResult() {
            this.state = null;
            this.bundle = null;
            this.productStateHash = null;
            this.taintedResources = null;
        }

        public PreBakeResult(PreBakeState state, String bundle, String productStateHash, List<TaintedResource> taintedResources) {
            this.state = state;
            this.bundle = bundle;
            this.productStateHash = productStateHash;
            this.taintedResources = Collections.unmodifiableList(
                    taintedResources.stream()
                            .map(TaintedResourceREST::new)
                            .collect(Collectors.toList())

            );
        }
    }

    public static class TaintedResourceREST {
        public final String url;
        public final String name;
        public final List<String> errors;

        public TaintedResourceREST(TaintedResource tr) {
            this.url = tr.getUrl();
            this.name = tr.getFullName();
            errors = Collections.unmodifiableList(
                    tr.getPrebakeErrors().stream()
                            .map(PrebakeError::toString)
                            .collect(Collectors.toList())
            );
        }

    }

    private Response whenPreBakeIsEnabled(Supplier<Response> normalResponse) {
        if (config.isPreBakeEnabled()) {
            return normalResponse.get();
        } else {
            log.warn("Pre-baking called but feature is not enabled!");
            return status(Status.FORBIDDEN).build();
        }
    }
}
