package com.atlassian.webresource.plugin.prebake.discovery;

import com.atlassian.plugin.webresource.UrlMode;
import com.atlassian.plugin.webresource.WebResourceIntegration;
import com.atlassian.plugin.webresource.impl.config.Config;
import com.atlassian.plugin.webresource.prebake.PrebakeWebResourceAssemblerFactory;
import com.atlassian.util.concurrent.ThreadFactories;
import com.atlassian.webresource.plugin.prebake.resources.ResourceCollector;
import com.atlassian.webresource.plugin.prebake.util.ConcurrentUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.net.URI;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import static com.atlassian.util.concurrent.ThreadFactories.Type.DAEMON;
import static com.atlassian.webresource.plugin.prebake.discovery.PreBakeState.CANCELLED;
import static com.atlassian.webresource.plugin.prebake.discovery.PreBakeState.NOTSTARTED;
import static java.util.Arrays.asList;

/**
 * Pre-bake operation executor.
 *
 * @since v3.5.0
 */
public final class WebResourcePreBaker {

    private static final Logger log = LoggerFactory.getLogger(WebResourcePreBaker.class);

    private final ExecutorService pool;
    private final Lock readState;
    private final Lock writeState;
    private volatile PreBakeState state;
    private final WebResourceBatch batch;
    private final Config config;
    private final WebResourceIntegration webResourceIntegration;
    private final PrebakeWebResourceAssemblerFactory webResourceAssemblerFactory;
    private final ResourceCollector resourceCollector;

    public WebResourcePreBaker(
            Config config,
            WebResourceIntegration webResourceIntegration,
            PrebakeWebResourceAssemblerFactory webResourceAssemblerFactory,
            ResourceCollector resourceCollector) {
        this(
                config,
                webResourceIntegration,
                webResourceAssemblerFactory,
                new DefaultWebResourceBatch(webResourceAssemblerFactory.computeGlobalStateHash()),
                resourceCollector
        );
    }

    public WebResourcePreBaker(
            Config config,
            WebResourceIntegration webResourceIntegration,
            PrebakeWebResourceAssemblerFactory webResourceAssemblerFactory,
            WebResourceBatch batch,
            ResourceCollector resourceCollector) {
        ReentrantReadWriteLock rwl = new ReentrantReadWriteLock();
        this.readState = rwl.readLock();
        this.writeState = rwl.writeLock();
        this.pool = Executors.newSingleThreadExecutor(ThreadFactories.named("prebaker").type(DAEMON).build());
        this.state = NOTSTARTED;
        this.batch = batch;
        this.config = config;
        this.webResourceIntegration = webResourceIntegration;
        this.webResourceAssemblerFactory = webResourceAssemblerFactory;
        this.resourceCollector = resourceCollector;
    }

    public void start(WebResourceCrawler... crawlers) {
        start(asList(crawlers));
    }

    /**
     * Starts web-resource discovery.
     *
     * @param crawlers - list of web-resource crawlers.
     */
    public void start(List<WebResourceCrawler> crawlers) {
        pool.execute(new DiscoveryTask(this, batch, crawlers, resourceCollector));
    }

    public PreBakeState getState() {
        readState.lock();
        try {
            return state;
        } finally {
            readState.unlock();
        }
    }

    /**
     * @return true if state was changed, otherwise false.
     */
    public boolean setState(PreBakeState newState) {
        writeState.lock();
        try {
            if (state == newState) {
                return false;
            } else {
                if (CANCELLED == newState) {
                    ConcurrentUtil.stopExecutor(pool, 200);
                }
                state = newState;
                return true;
            }
        } finally {
            writeState.unlock();
        }
    }

    /**
     * @return true if task was canceled and false if it wasn't in RUNNING state.
     */
    public boolean cancel() {
        return setState(CANCELLED);
    }

    public WebResourceBatch getBatch() {
        return batch;
    }

    public File getBundleDir() {
        return webResourceIntegration.getTemporaryDirectory();
    }

    public PrebakeWebResourceAssemblerFactory getWebResourceAssemblerFactory() {
        return webResourceAssemblerFactory;
    }

    protected URI toURI(String uriString) {
        String absoluteBaseUrl = webResourceIntegration.getBaseUrl(UrlMode.ABSOLUTE);
        String relativeBaseUrl = webResourceIntegration.getBaseUrl(UrlMode.RELATIVE);

        if (uriString.startsWith(absoluteBaseUrl)) {
            return URI.create(uriString);
        } else if (uriString.startsWith(relativeBaseUrl)) {
            return URI.create(absoluteBaseUrl.replace(relativeBaseUrl, "") + uriString);
        } else {
            return URI.create(absoluteBaseUrl + uriString);
        }
    }

    protected String relativeUrl(String uriString) {
        String baseUrl = webResourceIntegration.getBaseUrl();
        if (uriString.contains(baseUrl)) {
            return uriString.replace(baseUrl, "");
        } else {
            return uriString;
        }
    }

    public boolean isCSSPrebakingEnabled() {
        return config.isCSSPrebakingEnabled();
    }

}
