/**
 * This adapter is needed because JDK internals are strongly encapsulated since Java 17
 * and Gson can't rely on reflection anymore to parse Throwable class
 *
 * https://openjdk.org/jeps/403
 * https://github.com/google/gson/issues/2352#issuecomment-1481864208
 */

package com.atlassian.plugins.osgi.test.rest;

import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public class ThrowableTypeAdapter extends TypeAdapter<Throwable> {
    @Override
    public Throwable read(JsonReader reader) throws IOException {
        String detailMessage = "";
        StackTraceElement[] stackTrace = new StackTraceElement[0];
        
        reader.beginObject();
        while (reader.hasNext()) {
            String name = reader.nextName();
            if (name.equals("detailMessage")) {
                detailMessage = reader.nextString();
            } else if (name.equals("stackTrace")) {
                stackTrace = readStackTraceElements(reader);
            } else {
                reader.skipValue();
            }
        }
        reader.endObject();

        Throwable throwable = new Throwable(detailMessage);
        throwable.setStackTrace(stackTrace);

        return throwable;
    }

    private StackTraceElement[] readStackTraceElements(JsonReader reader) throws IOException {
        List<StackTraceElement> stackTraceElements = new ArrayList<>();

        reader.beginArray();
        while (reader.hasNext()) {
            stackTraceElements.add(readStackTraceElement(reader));
        }
        reader.endArray();

        return stackTraceElements.toArray(new StackTraceElement[0]);
    }

    private StackTraceElement readStackTraceElement(JsonReader reader) throws IOException {
        String declaringClass = "";
        String methodName = "";
        String fileName = "";
        int lineNumber = -1;

        reader.beginObject();
        while (reader.hasNext()) {
            String name = reader.nextName();
            if (name.equals("declaringClass")) {
                declaringClass = reader.nextString();
            } else if (name.equals("methodName")) {
                methodName = reader.nextString();
            } else if (name.equals("fileName")) {
                fileName = reader.nextString();
            } else if (name.equals("lineNumber")) {
                lineNumber = reader.nextInt();
            } else {
                reader.skipValue();
            }
        }
        reader.endObject();

        return new StackTraceElement(declaringClass, methodName, fileName, lineNumber);
    }

    @Override
    public void write(JsonWriter out, Throwable value) throws IOException {
        if (value == null) {
            out.nullValue();
            return;
        }

        out.beginObject();
        // Include exception type name to give more context; for example NullPointerException might
        // not have a message
        out.name("type");
        out.value(value.getClass().getSimpleName());

        out.name("message");
        out.value(value.getMessage());

        Throwable cause = value.getCause();
        if (cause != null) {
            out.name("cause");
            write(out, cause);
        }

        Throwable[] suppressedArray = value.getSuppressed();
        if (suppressedArray.length > 0) {
            out.name("suppressed");
            out.beginArray();

            for (Throwable suppressed : suppressedArray) {
                write(out, suppressed);
            }

            out.endArray();
        }
        out.endObject();
    }
}
