package com.atlassian.plugin.descriptors;

import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.ModuleDescriptorFactory;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import org.dom4j.Element;

/**
 * Utility class to create UnloadableModuleDescriptor instances when there are problems
 */
public final class UnloadableModuleDescriptorFactory {
    /**
     * Creates a new UnloadableModuleDescriptor, for when a problem occurs during the construction
     * of the ModuleDescriptor itself.
     * <p>
     * This instance has the same information as the original ModuleDescriptor, but also contains
     * an error message that reports the error.
     *
     * @param plugin                  the Plugin the ModuleDescriptor belongs to
     * @param element                 the XML Element used to construct the ModuleDescriptor
     * @param e                       the Throwable
     * @param moduleDescriptorFactory a ModuleDescriptorFactory used to retrieve ModuleDescriptor instances
     * @return a new UnloadableModuleDescriptor instance
     * @throws PluginParseException if there was a problem constructing the UnloadableModuleDescriptor
     * @deprecated since 7.5.0, use {@link #createUnloadableModuleDescriptor(Plugin, com.atlassian.plugin.module.Element, Throwable, ModuleDescriptorFactory)}
     * instead as this will be removed in Atlassian Plugins 8.0.0.
     */
    @Deprecated
    public static UnloadableModuleDescriptor createUnloadableModuleDescriptor(final Plugin plugin, final Element element, final Throwable e, final ModuleDescriptorFactory moduleDescriptorFactory) {
        return initNoOpModuleDescriptor(new UnloadableModuleDescriptor(), plugin, element, e, moduleDescriptorFactory);
    }

    /**
     * Creates a new UnloadableModuleDescriptor, for when a problem occurs during the construction
     * of the ModuleDescriptor itself.
     * <p>
     * This instance has the same information as the original ModuleDescriptor, but also contains
     * an error message that reports the error.
     *
     * @param plugin                  the Plugin the ModuleDescriptor belongs to
     * @param element                 the Element used to construct the ModuleDescriptor
     * @param e                       the Throwable
     * @param moduleDescriptorFactory a ModuleDescriptorFactory used to retrieve ModuleDescriptor instances
     * @return a new UnloadableModuleDescriptor instance
     * @throws PluginParseException if there was a problem constructing the UnloadableModuleDescriptor
     */
    public static UnloadableModuleDescriptor createUnloadableModuleDescriptor(final Plugin plugin, final com.atlassian.plugin.module.Element element, final Throwable e, final ModuleDescriptorFactory moduleDescriptorFactory) {
        return initNoOpModuleDescriptor(new UnloadableModuleDescriptor(), plugin, element, e, moduleDescriptorFactory);
    }

    /**
     * @deprecated since 7.5.0, use {@link #initNoOpModuleDescriptor(AbstractNoOpModuleDescriptor, Plugin, com.atlassian.plugin.module.Element, Throwable, ModuleDescriptorFactory)}
     * instead as this will be removed in Atlassian Plugins 8.0.0.
     */
    @Deprecated
    public static <T extends AbstractNoOpModuleDescriptor> T initNoOpModuleDescriptor(T descriptor, Plugin plugin, Element element, Throwable e, ModuleDescriptorFactory moduleDescriptorFactory) {
        descriptor.init(plugin, element);

        final String name = element.getName();
        final Class<? extends ModuleDescriptor> descriptorClass = moduleDescriptorFactory.getModuleDescriptorClass(name);
        String descriptorClassName;

        if (descriptorClass == null) {
            descriptorClassName = descriptor.getKey();
        } else {
            descriptorClassName = descriptorClass.getName();
        }

        final String errorMsg = constructErrorMessage(name, descriptorClassName, e);

        descriptor.setErrorText(errorMsg);

        return descriptor;
    }

    public static <T extends AbstractNoOpModuleDescriptor> T initNoOpModuleDescriptor(T descriptor, Plugin plugin, com.atlassian.plugin.module.Element element, Throwable e, ModuleDescriptorFactory moduleDescriptorFactory) {
        return initNoOpModuleDescriptor(descriptor, plugin, element.getDelegate(), e, moduleDescriptorFactory);
    }

    /**
     * Creates a new UnloadableModuleDescriptor based on an existing ModuleDescriptor, descriptor
     * <p>
     * This method uses the information in an existing descriptor to construct a new UnloadableModuleDescriptor.
     *
     * @param plugin     the Plugin the ModuleDescriptor belongs to
     * @param descriptor the ModuleDescriptor that reported an error
     * @param e          the Throwable
     * @return a new UnloadableModuleDescriptor instance
     */
    public static UnloadableModuleDescriptor createUnloadableModuleDescriptor(final Plugin plugin, final ModuleDescriptor<?> descriptor, final Throwable e) {
        final UnloadableModuleDescriptor unloadableDescriptor = new UnloadableModuleDescriptor();
        unloadableDescriptor.setName(descriptor.getName());
        unloadableDescriptor.setKey(descriptor.getKey());
        unloadableDescriptor.setPlugin(plugin);

        final String errorMsg = constructErrorMessage(descriptor.getDisplayName(),
                descriptor.getModuleClass() == null ? descriptor.getName() : descriptor.getModuleClass().getName(), e);

        unloadableDescriptor.setErrorText(errorMsg);

        return unloadableDescriptor;
    }

    /**
     * Constructs an error message from a module and exception
     *
     * @param moduleName  the name of the module
     * @param moduleClass the class of the module
     * @param e           the Throwable
     * @return an appropriate String representing the error
     */
    private static String constructErrorMessage(final String moduleName, final String moduleClass, final Throwable e) {
        String errorMsg;
        if (e instanceof PluginParseException) {
            errorMsg = "There was a problem loading the descriptor for module '" + moduleName + ".";
        } else if (e instanceof InstantiationException) {
            errorMsg = "Could not instantiate module descriptor: " + moduleClass + ".";
        } else if (e instanceof IllegalAccessException) {
            errorMsg = "Exception instantiating module descriptor: " + moduleClass + ".";
        } else if (e instanceof ClassNotFoundException) {
            errorMsg = "Could not find module descriptor class: " + moduleClass + ".";
        } else if (e instanceof NoClassDefFoundError) {
            errorMsg = "A required class was missing: " + moduleClass + ". Please check that you have all of the required dependencies.";
        } else {
            errorMsg = "There was a problem loading the module descriptor: " + moduleClass + ".";
        }
        return errorMsg + " " + e.getMessage();
    }
}
