define('atlassian-nps-plugin/js/nps/config-manager-server', [
    'atlassian-nps-plugin/js/amd-shims/ajs',
    'jquery',
    'atlassian-nps-plugin/js/nps/client-storage',
    'atlassian-nps-plugin/js/nps/submission'
],
/**
 * Server-side storage implementation of NPS Configuration. Instances store the configuration on the server,
 * passing the data via REST.
 * @param AJS
 * @param $ jQuery
 * @param ClientStorage client-side storage mechanism name spaced by user for retrieving scheduling data
 * @param Submission handles the firing of analytics events
 * @returns {object} Server-side NPS configuration implementation
 */
function (
    AJS,
    $,
    ClientStorage,
    Submission
) {
    "use strict";

    var SERVER_DATA = {
        optedOut: null,
        dismissedCount: null,
        timeToNextSurvey: null
    };

    return {
        getCachedData: function () {
            return SERVER_DATA;
        },

        removeCachedData: function () {
            SERVER_DATA = {
                optedOut: null,
                dismissedCount: null,
                timeToNextSurvey: null
            };
        },

        /**
         * Retrieves the NPS configuration from the server. This configuration includes the dismissedCount (int),
         * optedOut (boolean), and timeToNextSurvey (long). The configuration is cached locally to prevent additional
         * AJAX requests
         * @returns a promise that is fulfilled when the server request returns successfully
         */
        fetchConfig: function () {
            return $.ajax({
                url: AJS.contextPath() + "/rest/nps/1.0/config",
                type: "GET",
                contentType: "application/json",
                cache: false,
                success: function (result) {
                    if (result.dismissedCount === null) {
                        result.dismissedCount = 0;
                    }
                    SERVER_DATA = result;
                    ClientStorage.setProductValue("nextSurveyDate", Date.now() + SERVER_DATA.timeToNextSurvey);
                },
                error: function (jqXHR) {
                    AJS.warn("Error getting NPS config: ", jqXHR);
                }
            });
        },

        /**
         * Sets NPS configuration on the server. The arguments can either be key, value or a configuration object with
         * property name/value pairs. Upon successful acknowledgement by the server, the cached SERVER_DATA is updated
         * to reflect the changes
         * @returns a promise that is fulfilled when the server request returns successfully
         */
        setConfig: function () {
            var obj;
            if (typeof arguments[0] === "object") {
                obj = arguments[0];
            } else {
                if (arguments.length !== 2) {
                    throw ("Need to provide key/value as argument to set NPS server config");
                }
                obj = {};
                obj[arguments[0]] = arguments[1];
            }
            // The NPS plugin does not let us set a nextSurveyDate in the past. In order to allow tests
            // to modify this constraint, we are using a forceUpdate property to tell the REST that we
            // allow values to be set in the past.
            if (localStorage && localStorage.getItem('nps-testing') === "true") {
                obj.forceUpdate = true;
            }
            return $.ajax({
                url: AJS.contextPath() + "/rest/nps/1.0/config",
                type: "PUT",
                contentType: "application/json",
                data: JSON.stringify(obj),
                success: function () {
                    // If successfully synced to server, update the cached SERVER_DATA
                    $.extend(SERVER_DATA, obj);
                },
                error: function (jqXHR) {
                    AJS.warn("Error setting NPS config: ", jqXHR);
                }
            });
        },

        /**
         * Requests the server to schedule the next survey date. This is likely done when a user has finished completing
         * or dismissing a survey or upon the initial loading of the NPS plugin.
         * @returns a promise that is fulfilled when the server request returns successfully
         */
        scheduleNextSurveyDate: function () {
            return $.ajax({
                url: AJS.contextPath() + "/rest/nps/1.0/config/nextSurveyDate",
                type: "POST",
                contentType: "application/json",
                success: function(nextSurveyDate) {
                    Submission.scheduleNextSurveyDate({ nextSurveyDate: nextSurveyDate });
                    ClientStorage.setProductValue("nextSurveyDate", nextSurveyDate);
                },
                error: function (jqXHR) {
                    AJS.warn("Error scheduling next survey date for NPS: ", jqXHR);
                }
            });
        }
    };
});
