define('atlassian-nps-plugin/js/nps/client-storage', [
    'atlassian-nps-plugin/js/amd-shims/ajs',
    'atlassian-nps-plugin/js/nps/product'
],
/**
 * Wrapper for interactions with local storage.
 * @param AJS
 * @param Product adapter for a specific product such as JIRA or Confluence
 */
function (
    AJS,
    Product
) {
    "use strict";

    var storageKey;

    function getStoragePrefix() {
        if (storageKey === undefined) {
            storageKey = "nps-" + Product.getUserKey();
        }
        return storageKey;
    }

    function createLocalStorageKey(key) {
        return getStoragePrefix() + "-" + key;
    }

    function createProductStorageKey(key) {
        return createLocalStorageKey(Product.getProductName().toLowerCase().replace(/\s+/g, '') + "-" + key);
    }

    return {
        set: function (key, value) {
            try {
                localStorage.setItem(createLocalStorageKey(key), value);
            } catch (e) {
                // TODO: need better exception handling (https://jira.atlassian.com/browse/NPS-61)
                AJS.warn("Could not set NPS value: " + e);
            }
        },

        setProductValue: function (key, value) {
            try {
                localStorage.setItem(createProductStorageKey(key), value);
            } catch (e) {
                // TODO: need better exception handling (https://jira.atlassian.com/browse/NPS-61)
                AJS.warn("Could not set NPS value: " + e);
            }
        },

        get: function (key) {
            return localStorage.getItem(createLocalStorageKey(key));
        },

        getProductValue: function (key) {
            return localStorage.getItem(createProductStorageKey(key));
        },

        remove: function (key) {
            localStorage.removeItem(createLocalStorageKey(key));
        },

        removeProductValue: function (key) {
            localStorage.removeItem(createProductStorageKey(key));
        },

        containsKey: function (key) {
            return localStorage.getItem(createLocalStorageKey(key)) !== null;
        },

        getNumber: function (key) {
            var number = parseInt(this.get(key), 10);
            if (isNaN(number)) {
                return 0;
            }
            return number;
        },

        increment: function (key) {
            this.set(key, (this.getNumber(key) + 1));
        }
    };
});
