package com.atlassian.plugins.navlink.util.url;

import com.atlassian.sal.api.ApplicationProperties;
import org.apache.commons.lang.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.net.URI;

/**
 * Creates {@link String} instances containing a URL taking into account the application
 * s base url as defined in the {@link ApplicationProperties}.
 *
 * @see com.atlassian.sal.api.ApplicationProperties#getBaseUrl()
 *
 * @since v1.3
 */
public class UrlFactory
{
    private final ApplicationProperties applicationProperties;

    public UrlFactory(final ApplicationProperties applicationProperties)
    {
        this.applicationProperties = applicationProperties;
    }

    /**
     * <p>Creates an absolute url based on the contents of the specified initial url.</p>
     *
     * <p>If the input url is <code>null</code>, then null is returned.</p>
     *
     * <p>Otherwise, if the specified initial url is a relative url,
     * an absolute url is constructed by prefixing the host application's base url to the provided initial url.
     * On the other hand, is the specified initial url is already an absolute url, the initial url is returned.
     * </p>
     *
     * @param initialUrl The initial url to transform into an absolute url.
     * @return A {@link String} containing an absolute url for the specified initial url, or null if the provided
     * initial url is null.
     */
    public String toAbsoluteUrl(@Nullable final String initialUrl)
    {
        return toAbsoluteUrl(applicationProperties.getBaseUrl(), initialUrl);
    }

    @Nonnull
    public BaseUrl getBaseUrl()
    {
        return new BaseUrl(applicationProperties.getBaseUrl());
    }

    /**
     * @see #toAbsoluteUrl(String)
     */
    public static String toAbsoluteUrl(@Nullable final String baseUrl, @Nullable final String initialUrl)
    {
        if (baseUrl == null || initialUrl == null)
        {
            return null;
        }

        final String baseUrlWithoutTrailingSlash = StringUtils.stripEnd(baseUrl, "/");
        if (!(initialUrl.startsWith("http://") || initialUrl.startsWith("https://")))
        {
            return baseUrlWithoutTrailingSlash + initialUrl;
        }
        return initialUrl;
    }

    public String toRelativeUrlWithContextPath(String relativeUrl)
    {
        if (relativeUrl == null)
        {
            return null;
        }

        if (relativeUrl.startsWith("http://") || relativeUrl.startsWith("https://"))
        {
            return relativeUrl; //URL is actually an absolute URL - return as is
        }

        String baseUrl = getBaseUrl().getBaseUrl();
        URI uri = URI.create(baseUrl);
        return StringUtils.stripEnd(uri.getPath(), "/") + "/" + StringUtils.stripStart(relativeUrl, "/");
    }
}
