package com.atlassian.plugins.navlink.util.executor;

import com.atlassian.sal.api.executor.ThreadLocalDelegateExecutorFactory;
import com.atlassian.util.concurrent.ThreadFactories;
import com.atlassian.util.concurrent.ThreadFactories.Type;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import org.springframework.beans.factory.DisposableBean;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * A simple wrapper around an executor configured to use a thread pool that produces daemon threads. 
 *
 */
public class DaemonExecutorServiceImpl implements DaemonExecutorService, DisposableBean
{
    private final ScheduledExecutorService executor;

    public DaemonExecutorServiceImpl(ThreadLocalDelegateExecutorFactory delegateExecutorFactory)
    {
        ThreadFactory factory = ThreadFactories
                .namedThreadFactory("Navlink Plugin Executor", Type.DAEMON, THREAD_POOL_PRIORITY);

        final ScheduledThreadPoolExecutor delegate = new ScheduledThreadPoolExecutor(THREAD_POOL_SIZE, factory);
        delegate.setExecuteExistingDelayedTasksAfterShutdownPolicy(false);
        this.executor = delegateExecutorFactory.createScheduledExecutorService(delegate);
    }

    @Override
    public void execute(Runnable command)
    {
        executor.execute(command);
    }

    @Override
    public void shutdown()
    {
        executor.shutdown();
    }

    @Override
    public List<Runnable> shutdownNow()
    {
        return executor.shutdownNow();
    }

    @Override
    public boolean isShutdown()
    {
        return executor.isShutdown();
    }

    @Override
    public boolean isTerminated()
    {
        return executor.isTerminated();
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException
    {
        return executor.awaitTermination(timeout, unit);
    }

    @Override
    public <T> Future<T> submit(Callable<T> task)
    {
        return executor.submit(task);
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result)
    {
        return executor.submit(task, result);
    }

    @Override
    public Future<?> submit(Runnable task)
    {
        return executor.submit(task);
    }

    @Override
    public ScheduledFuture<?> schedule(Runnable command, long delay, TimeUnit unit)
    {
        return executor.schedule(command, delay, unit);
    }

    @Override
    public <V> ScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit unit)
    {
        return executor.schedule(callable, delay, unit);
    }

    @Override
    public ScheduledFuture<?> scheduleAtFixedRate(Runnable command, long initialDelay, long period, TimeUnit unit)
    {
        return executor.scheduleAtFixedRate(command, initialDelay, period, unit);
    }

    @Override
    public ScheduledFuture<?> scheduleWithFixedDelay(Runnable command, long initialDelay, long delay, TimeUnit unit)
    {
        return executor.scheduleAtFixedRate(command, initialDelay, delay, unit);
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException
    {
        return executor.invokeAll(tasks);
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
            throws InterruptedException
    {
        return executor.invokeAll(tasks, timeout, unit);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException
    {
        return executor.invokeAny(tasks);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
            throws InterruptedException, ExecutionException, TimeoutException
    {
        return executor.invokeAny(tasks, timeout, unit);
    }

    @Override
    public <O, I extends Iterable<O>> ImmutableList<O> invokeAllAndGet(Iterable<? extends Callable<I>> callables, long timeout, TimeUnit unit) throws ExecutionException, InterruptedException
    {
        ImmutableList.Builder<O> builder = ImmutableList.builder();

        List<Future<I>> futures = executor.invokeAll(Lists.newArrayList(callables), timeout, unit);
        for (Future<I> future : futures)
        {
            if(!future.isCancelled())
                builder.addAll(future.get());
        }
       
        return builder.build();
    }

    @Override
    public void destroy() throws Exception
    {
        executor.shutdown();
    }
}
