package com.atlassian.plugins.navlink.util.date;

import com.google.common.base.Function;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

/**
 * <p/>
 * Uses the <a href="http://www.w3.org/TR/NOTE-datetime">ISO-8601 date format</a> to format
 * dates in a consistent way useful for remote communication.
 *
 * @since 3.0
 */
public final class UniversalDateFormatter
{
    private static final DateTimeFormatter FORMATTER = ISODateTimeFormat.dateTimeNoMillis();

    private UniversalDateFormatter()
    {
        throw new AssertionError("Don't instantiate me");
    }

    /**
     * Format the date according to the ISO-8601 (complete version up to seconds), using provided <tt>timeZone</tt>.
     *
     * @param date date to format
     * @param timeZone target time zone
     * @return ISO-8801-formatted date
     */
    public static String format(DateTime date, DateTimeZone timeZone)
    {
        return FORMATTER.withZone(timeZone).print(date);
    }

    /**
     * Format the date according to the ISO-8601 (complete version up to seconds), using UTC time zone, which
     * is generally a recommended practice unless the remote client is interested in the time zone associated
     * with given context.
     *
     * @param date date to format
     * @return ISO-8801-formatted date
     */
    public static String formatUtc(DateTime date)
    {
        return format(date, DateTimeZone.UTC);
    }


    /**
     * Parse a date assuming it was formatted according to ISO-8601 (complete version up to seconds, including time zone
     * mark).
     *
     * @param date string date to parse
     * @return parsed date, in local time zone
     */
    public static DateTime parse(String date)
    {
        return FORMATTER.parseDateTime(date);
    }

    public static Function<String,DateTime> parseFunction()
    {
        return new Function<String, DateTime>()
        {
            @Override
            public DateTime apply(String from)
            {
                return parse(from);
            }
        };
    }

}
