package com.atlassian.plugins.navlink.producer.navigation.rest;

import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.services.LocalNavigationLinkService;
import com.atlassian.plugins.navlink.util.JsonStringEncoder;
import com.atlassian.plugins.navlink.util.LastModifiedFormatter;
import com.atlassian.plugins.navlink.util.url.BaseUrl;
import com.atlassian.plugins.navlink.util.url.SelfUrl;
import com.atlassian.plugins.navlink.util.url.UrlFactory;
import com.atlassian.sal.api.message.LocaleResolver;
import com.atlassian.templaterenderer.TemplateRenderer;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.core.HttpHeaders;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import static com.atlassian.plugins.navlink.util.CacheControlFactory.withConfiguredMaxAgeAndStaleContentExtension;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static org.apache.commons.lang.CharEncoding.UTF_8;

/**
 * Implements the navigation capabilities REST endpoint.
 *
 * @since 2.0.3
 */
public class NavigationServlet extends HttpServlet
{
    static final String INIT_PARAM_CAPABILITIES_REST_ENDPOINT = "capabilitiesRestEndpoint";
    private static final String RESPONSE_TEMPLATE = "templates/navigation.vm";

    private final Logger logger = LoggerFactory.getLogger(NavigationServlet.class);
    private final LocalNavigationLinkService localNavigationLinkService;
    private final TemplateRenderer templateRenderer;
    private final UrlFactory urlFactory;
    private final LocaleResolver localeResolver;

    private String capabilitiesRestEndpoint;

    public NavigationServlet(final LocalNavigationLinkService localNavigationLinkService, final TemplateRenderer templateRenderer,
                             final UrlFactory urlFactory, final LocaleResolver localeResolver)
    {
        this.localNavigationLinkService = localNavigationLinkService;
        this.templateRenderer = templateRenderer;
        this.urlFactory = urlFactory;
        this.localeResolver = localeResolver;
    }

    @Override
    public void init() throws ServletException
    {
        capabilitiesRestEndpoint = getInitParameter(INIT_PARAM_CAPABILITIES_REST_ENDPOINT);
        if (Strings.isNullOrEmpty(capabilitiesRestEndpoint))
        {
            throw new ServletException("init param not specified or empty: '" + INIT_PARAM_CAPABILITIES_REST_ENDPOINT + "'");
        }
    }

    @Override
    protected void doGet(final HttpServletRequest httpServletRequest, final HttpServletResponse httpServletResponse) throws ServletException, IOException
    {
        try
        {
            httpServletResponse.setContentType(APPLICATION_JSON);
            httpServletResponse.setCharacterEncoding(UTF_8);
            httpServletResponse.setHeader(HttpHeaders.CACHE_CONTROL, withConfiguredMaxAgeAndStaleContentExtension().toString());
            httpServletResponse.setHeader(HttpHeaders.LAST_MODIFIED, LastModifiedFormatter.formatCurrentTimeMillis());

            final Map<String, Object> context = createContext(httpServletRequest);
            final PrintWriter writer = httpServletResponse.getWriter();
            renderTemplate(context, writer);
        }
        catch (IOException e)
        {
            handleException(httpServletResponse, e);
        }
    }

    private void renderTemplate(@Nonnull final Map<String, Object> context, @Nonnull final Writer writer) throws IOException
    {
        templateRenderer.render(RESPONSE_TEMPLATE, context, writer);
    }

    private Map<String, Object> createContext(@Nonnull final HttpServletRequest httpServletRequest)
    {
        final BaseUrl baseUrl = urlFactory.getBaseUrl();
        final String selfUrl = SelfUrl.extractFrom(httpServletRequest);
        final Locale locale = getLocaleFromRequest(httpServletRequest);
        final Set<NavigationLink> navigationLinks = localNavigationLinkService.all(locale);
        final Map<MenuItemKey, List<NavigationLink>> navigationLinksGroupedByMenuItemKey = groupNavigationLinksByKey(navigationLinks);
        return ImmutableMap.<String, Object>builder()
                .put("baseUrl", baseUrl)
                .put("selfUrl", selfUrl)
                .put("collectionUrl", capabilitiesRestEndpoint)
                .put("languageTag", LanguageParameter.encodeValue(locale))
                .put("navigationLinks", navigationLinksGroupedByMenuItemKey)
                .put("json", new JsonStringEncoder())
                .build();
    }

    @Nonnull
    private Locale getLocaleFromRequest(@Nonnull final HttpServletRequest httpServletRequest)
    {
        final Locale locale = LanguageParameter.extractFrom(httpServletRequest, Locale.getDefault());
        return localeResolver.getSupportedLocales().contains(locale) ? locale : Locale.getDefault();
    }

    @Nonnull
    private Map<MenuItemKey, List<NavigationLink>> groupNavigationLinksByKey(@Nonnull final Set<NavigationLink> navigationLinks)
    {
        final Map<MenuItemKey, List<NavigationLink>> links = new HashMap<MenuItemKey, List<NavigationLink>>();
        for (final NavigationLink navigationLink : navigationLinks)
        {
            getOrCreateList(links, new MenuItemKey(navigationLink.getKey())).add(navigationLink);
        }
        return links;
    }

    @Nonnull
    private <T> List<T> getOrCreateList(@Nonnull final Map<MenuItemKey, List<T>> map, @Nonnull final MenuItemKey mapKey)
    {
        List<T> navigationLinkEntityList = map.get(mapKey);
        if (navigationLinkEntityList == null)
        {
            navigationLinkEntityList = new LinkedList<T>();
            map.put(mapKey, navigationLinkEntityList);
        }
        return navigationLinkEntityList;
    }


    private void handleException(@Nonnull final HttpServletResponse httpServletResponse, @Nonnull final Exception e)
    {
        logger.warn("Failed to serialize navigation items: {}", e.getMessage());
        logger.debug("Stacktrace:", e);
        httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
    }
}
