package com.atlassian.plugins.navlink.producer.navigation.plugin;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugins.navlink.producer.navigation.services.RawNavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.services.NavigationLinkRepository;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Provides access to all navigation links defined through the plugin module type.
 *
 * @see NavigationLinkModuleDescriptor
 */
public class PluginModuleTypeBasedNavigationLinkRepository implements NavigationLinkRepository
{
    private final PluginAccessor pluginAccessor;

    public PluginModuleTypeBasedNavigationLinkRepository(@Nonnull final PluginAccessor pluginAccessor)
    {
        this.pluginAccessor = checkNotNull(pluginAccessor);
    }

    @Override
    public Iterable<RawNavigationLink> matching(@Nullable final Predicate<RawNavigationLink> criteria)
    {
        final List<NavigationLinkModuleDescriptor> enabledModuleDescriptors = getEnabledModuleDescriptors();
        final Iterable<RawNavigationLink> navigationLinkEntities = convertToLinks(enabledModuleDescriptors);
        final Iterable<RawNavigationLink> result = filterByCriteria(criteria, navigationLinkEntities);
        return Lists.newArrayList(result);
    }

    @Override
    public Iterable<RawNavigationLink> all()
    {
        return matching(null);
    }

    @Nonnull
    private List<NavigationLinkModuleDescriptor> getEnabledModuleDescriptors()
    {
        final List<NavigationLinkModuleDescriptor> result = pluginAccessor.getEnabledModuleDescriptorsByClass(NavigationLinkModuleDescriptor.class);
        return result != null ? result : Collections.<NavigationLinkModuleDescriptor>emptyList();
    }

    @Nonnull
    private Iterable<RawNavigationLink> convertToLinks(@Nonnull final List<NavigationLinkModuleDescriptor> enabledModuleDescriptors)
    {
        return Iterables.transform(enabledModuleDescriptors, new Function<NavigationLinkModuleDescriptor, RawNavigationLink>()
        {
            @Override
            public RawNavigationLink apply(@Nullable final NavigationLinkModuleDescriptor input)
            {
                return input != null ? input.getModule() : null;
            }
        });
    }

    @Nonnull
    private Iterable<RawNavigationLink> filterByCriteria(@Nullable final Predicate<RawNavigationLink> criteria, @Nonnull final Iterable<RawNavigationLink> navigationLinkEntities)
    {
        return criteria != null ? Iterables.filter(navigationLinkEntities, criteria) : navigationLinkEntities;
    }
}
