package com.atlassian.plugins.navlink.producer.navigation.plugin;

import com.atlassian.plugin.module.ModuleFactory;
import com.atlassian.plugin.util.validation.ValidationPattern;
import com.atlassian.plugin.web.WebInterfaceManager;
import com.atlassian.plugin.web.descriptors.AbstractWebFragmentModuleDescriptor;
import com.atlassian.plugins.navlink.producer.capabilities.services.ApplicationTypeService;
import com.atlassian.plugins.navlink.producer.navigation.links.LinkSource;
import com.atlassian.plugins.navlink.producer.navigation.services.RawNavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.services.RawNavigationLinkBuilder;
import com.atlassian.plugins.navlink.spi.weights.ApplicationWeights;
import com.google.common.base.Strings;
import org.dom4j.Element;

import javax.annotation.Nullable;

import static com.atlassian.plugin.util.validation.ValidationPattern.test;
import static com.google.common.base.Preconditions.checkNotNull;

public class NavigationLinkModuleDescriptor extends AbstractWebFragmentModuleDescriptor<RawNavigationLink>
{
    private volatile boolean enabled = false;

    private final ApplicationWeights applicationWeights;
    private final ApplicationTypeService applicationTypeService;

    public NavigationLinkModuleDescriptor(final ModuleFactory moduleClassFactory,
                                          final WebInterfaceManager webInterfaceManager,
                                          final ApplicationWeights applicationWeights,
                                          final ApplicationTypeService applicationTypeService)
    {
        super(checkNotNull(moduleClassFactory), checkNotNull(webInterfaceManager));
        this.applicationWeights = applicationWeights;
        this.applicationTypeService = applicationTypeService;
    }

    @Override
    protected void provideValidationRules(ValidationPattern pattern)
    {
        super.provideValidationRules(pattern);
        pattern.rule(
                test("@menu-key").withError("menu-key attribute is mandatory"),
                test("not(@menu-key) or string-length(normalize-space(@menu-key)) > 0").withError("menu-key is empty"),
                test("link").withError("link tag is mandatory"),
                test("string-length(normalize-space(link)) > 0").withError("link tag requires a link as content"),
                test("label").withError("label tag is mandatory"),
                test("string-length(normalize-space(label/@key)) > 0").withError("label tag requires a key attribute"),
                test("not(tooltip) or string-length(normalize-space(tooltip/@key)) > 0").withError("tooltip tag requires a key attribute"),
                test("not(icon) or string-length(normalize-space(icon)) > 0").withError("icon tag requires an image url as content"),
                test("not(application-type) or string-length(normalize-space(application-type)) > 0").withError("application-type requires the type name as content"),
                test("not(@weight) or string-length(normalize-space(@weight)) > 0").withError("weight attribute must have a value"),
                test("not(@weight) or number(@weight) = @weight").withError("weight attribute must be a number")
        );
    }

    @Override
    public void enabled()
    {
        super.enabled();
        enabled = true;
    }

    @Override
    public void disabled()
    {
        super.disabled();
        enabled = false;
    }

    @Override
    public RawNavigationLink getModule()
    {
        return enabled ? parseNavigationLinkEntity() : null;
    }

    private RawNavigationLink parseNavigationLinkEntity()
    {
        return new RawNavigationLinkBuilder()
                .key(parseMenuKey(element))
                .href(parseLink(element.element("link")))
                .labelKey(parseKeyAttributeFromElement(element.element("label")))
                .tooltipKey(parseKeyAttributeFromElement(element.element("tooltip")))
                .iconUrl(parseWebIcon(element.element("icon")))
                .weight(parseWeight())
                .applicationType(parseApplicationType())
                .self(isTrue(parseAttribute(element, "self")))
                .source(LinkSource.localDefault())
                .build();
    }

    private String parseMenuKey(final Element navigationLinkElement)
    {
        return parseAttribute(navigationLinkElement, "menu-key");
    }

    private String parseLink(@Nullable final Element linkElement)
    {
        return parseElementContent(linkElement);
    }

    private String parseKeyAttributeFromElement(@Nullable final Element tagElement)
    {
        return Strings.emptyToNull(parseAttribute(tagElement, "key"));
    }

    private String parseWebIcon(@Nullable final Element iconElement)
    {
        if (iconElement == null)
        {
            return null;
        }
        else
        {
            return parseElementContent(iconElement);
        }
    }

    private int parseWeight()
    {
        if (element.attributeValue("weight") != null)
        {
            return Integer.parseInt(parseAttribute(element, "weight"));
        }
        else
        {
            // default - host application's weights
            return applicationWeights.getApplicationWeight();
        }
    }

    private String parseApplicationType()
    {
        String applicationType = parseElementContent(element.element("application-type"));
        return Strings.isNullOrEmpty(applicationType) ? applicationTypeService.get() : applicationType;
    }

    private String parseElementContent(@Nullable final Element element)
    {
        final String elementContent = element != null ? element.getTextTrim() : null;
        return Strings.nullToEmpty(elementContent).trim();
    }

    private String parseAttribute(@Nullable final Element element, final String attributeName)
    {
        final String attributeValue = element != null ? element.attributeValue(attributeName) : null;
        return Strings.nullToEmpty(attributeValue).trim();
    }

    private boolean isTrue(String val)
    {
        return Boolean.TRUE.toString().equalsIgnoreCase(val);
    }

}
