package com.atlassian.plugins.navlink.producer.navigation.links;

import com.atlassian.plugins.navlink.producer.capabilities.services.ApplicationTypeService;
import com.google.common.base.Objects;
import org.joda.time.DateTime;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Encapsulates universal core attributes of navigation links. Designed for inheritance by immutable data transport
 * classes representing various flavours of navigation links.
 *
 * @since 2.2.3
 */
public abstract class NavigationLinkBase
{
    /**
     * Value to use for the key for custom links
     */
    public static final String CUSTOM_APPS_KEY = "custom-apps";

    private final String key;
    private final String href;
    private final String baseUrl;
    private final String iconUrl;

    private final LinkSource source;
    private final int weight;
    private final boolean self;

    // TODO: those 2 guys should go into LinkSource... perhaps we should just add appwithproperties to the link here
    private final String applicationType;
    private final DateTime buildDate;

    private NavigationLinkBase(@Nonnull String key, @Nonnull String href, String baseUrl, String iconUrl,
                               String applicationType, LinkSource source, int weight, boolean self, DateTime buildDate)
    {
        this.key = checkNotNull(key, "key");
        this.href = checkNotNull(href, "href");
        this.baseUrl = baseUrl;
        this.iconUrl = iconUrl;
        this.applicationType = applicationType != null ? applicationType : ApplicationTypeService.DEFAULT_APPLICATION_TYPE;
        this.source = source != null ? source : LinkSource.unknown();
        this.weight = weight;
        this.self = self;
        this.buildDate = buildDate;
    }

    protected NavigationLinkBase(NavigationLinkBuilderBase<?,?> builder)
    {
        this(builder.key, builder.href, builder.baseUrl, builder.iconUrl, builder.applicationType, builder.source, builder.weight,
                builder.self, builder.buildDate);
    }


    @Nonnull
    public final String getKey()
    {
        return key;
    }

    @Nonnull
    public final String getHref()
    {
        return href;
    }

    @Nullable
    public final String getBaseUrl()
    {
        return baseUrl;
    }

    @Nullable
    public final String getIconUrl()
    {
        return iconUrl;
    }

    @Nonnull
    public final String getApplicationType()
    {
        return applicationType;
    }

    @Nonnull
    public DateTime getBuildDate()
    {
        return buildDate;
    }

    /**
     * Source of this link.
     *
     * @return source hinting at where the link comes from.
     * @see LinkSource
     */
    @Nonnull
    public final LinkSource getSource()
    {
        return source;
    }


    /**
     * A weight assigned to this link.
     *
     * @return weight of this navigation link
     */
    public final int weight()
    {
        return weight;
    }

    /**
     * Whether this link points to application we are currently running in.
     *
     * @return if <code>true</code>, this is a link to the current host application
     */
    public final boolean isSelf()
    {
        return self;
    }

    /**
     * For use in subclasses' implementations of {@link #equals(Object)}.
     *
     * @param that the object to check equality against
     * @return <code>true</code>, if properties of this link base are equal to link base properties of
     * <tt>that</tt> (which by itself <strong>does not</strong> suffice to determine whether both instances are equal).
     * Notably, this method does not check whether both objects are instances of the same class.
     */
    protected final boolean isEqualTo(NavigationLinkBase that)
    {
        return Objects.equal(this.key, that.key) &&
                Objects.equal(this.href, that.href) &&
                Objects.equal(this.baseUrl, that.baseUrl) &&
                Objects.equal(this.iconUrl, that.iconUrl) &&
                Objects.equal(this.applicationType, that.applicationType) &&
                Objects.equal(this.source, that.source) &&
                Objects.equal(this.weight, that.weight) &&
                Objects.equal(this.self, that.self) &&
                Objects.equal(this.buildDate, that.buildDate);
    }

    /**
     * For use in subclasses' implementations of {@link #hashCode()}.
     *
     * @return hash code of all fields in this instance that are significant to the equivalence relation.
     *
     * @see #isEqualTo(NavigationLinkBase)
     */
    protected final int hashCodeBase()
    {
        return Objects.hashCode(this.key, this.href, this.baseUrl, this.iconUrl, this.applicationType, this.source,
                this.weight, this.self, this.buildDate);
    }

    /**
     * Produces a string of key-value pairs representing property names and their respective values in this
     * instance. For use in subclasses' implementations of {@link #toString()}.
     *
     * @return a string representation of this object suitable for use in {@link #toString()} implementations.
     */
    protected final String toStringBase()
    {
        return  "key='" + key + '\'' +
                ", href='" + href + '\'' +
                ", baseUrl='" + this.baseUrl + '\'' +
                ", iconUrl='" + this.iconUrl + '\'' +
                ", applicationType='" + this.applicationType + '\'' +
                ", source='" + this.source + '\'' +
                ", weight='" + this.weight + '\'' +
                ", self='" + this.self + '\'' +
                ", buildDate='" + this.buildDate + '\'';
    }

}
