package com.atlassian.plugins.navlink.producer.contentlinks.rest;

import com.atlassian.plugin.web.model.WebLabel;
import com.atlassian.plugin.web.model.WebLink;
import com.atlassian.plugins.navlink.producer.contentlinks.customcontentlink.CustomContentLink;
import com.atlassian.plugins.navlink.producer.contentlinks.plugin.ContentLinkModuleDescriptor;
import com.atlassian.plugins.navlink.util.url.UrlFactory;
import com.google.common.base.Function;
import com.google.common.collect.Lists;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import java.util.List;
import java.util.Map;

public class ContentLinkEntityFactory
{
    private UrlFactory urlFactory;
    private Map<String, Object> context;
    private HttpServletRequest request;

    public ContentLinkEntityFactory(HttpServletRequest request, Map<String, Object> context, UrlFactory urlFactory)
    {
        this.urlFactory = urlFactory;
        this.request = request;
        this.context = context;
    }

    public List<ContentLinkEntity> create(List<ContentLinkModuleDescriptor> moduleDescriptors)
    {
        return Lists.transform(moduleDescriptors,
                new Function<ContentLinkModuleDescriptor, ContentLinkEntity>()
                {
                    @Override
                    public ContentLinkEntity apply(@Nullable ContentLinkModuleDescriptor input)
                    {
                        return create(input);
                    }
                });
    }

    public ContentLinkEntity create(ContentLinkModuleDescriptor contentLinkModuleDescriptor)
    {
        if (contentLinkModuleDescriptor != null)
        {
            String label = "";
            final WebLabel webLabel = contentLinkModuleDescriptor.getWebLabel();
            if (webLabel != null)
            {
                label = webLabel.getDisplayableLabel(request, context);
            }

            String absoluteUrlLink = "";
            final WebLink webLink = contentLinkModuleDescriptor.getLink();
            if (webLink != null)
            {
                absoluteUrlLink = urlFactory.toAbsoluteUrl(webLink.getRenderedUrl(context));
            }

            String tooltip = "";
            final WebLabel tooltipLabel = contentLinkModuleDescriptor.getTooltip();
            if (tooltipLabel != null)
            {
                tooltip = tooltipLabel.getDisplayableLabel(request, context);
            }

            return new ContentLinkEntity(absoluteUrlLink, label, tooltip, false);

        }
        return null;
    }

    /**
     * Transforms a list of CustomContentLinks into a list of ContentLinkEntity objects for rendering by a REST endpoint.
     * @see ContentLinkEntityFactory.createFromCustomContentLink
     * @param customContentLinks a List of CustomContentLinks to transform
     * @param convertToAbsolute determines the transformation that should be applied to the URL in the entity.
     *                          If true, the entity URL should be converted to an absolute URL by appending the full base
     *                          path including context path.
     *                          If false, the URL rendered will be converted to a relative URL that includes the context path.
     * @return a transformed list of ContentLinkEntity objects
     */
    public List<ContentLinkEntity> createFromCustomContentLinks(List<CustomContentLink> customContentLinks,
                                                                final boolean convertToAbsolute)
    {
        return Lists.transform(customContentLinks,
                new Function<CustomContentLink, ContentLinkEntity>()
                {
                    @Override
                    public ContentLinkEntity apply(@Nullable CustomContentLink input)
                    {
                        return createFromCustomContentLink(input, convertToAbsolute);
                    }
                });
    }

    /**
     * Transforms a CustomContentLink provided by a product into a ContentLinkEntity for rendering by a REST endpoint
     * @param entity the CustomContentLink to transform
     * @param convertToAbsolute determines the transformation that should be applied to the URL in the entity.
     *                          If true, the entity URL should be converted to an absolute URL by appending the full base
     *                          path including context path.
     *                          If false, the URL rendered will be converted to a relative URL that includes the context path.
     * @return the ContentLinkEntity for the REST endpoint to render
     */
    public ContentLinkEntity createFromCustomContentLink(CustomContentLink entity, boolean convertToAbsolute)
    {
        if (convertToAbsolute)
        {
            return new ContentLinkEntity(urlFactory.toAbsoluteUrl(entity.getLinkUrl()), entity.getLinkLabel(), "", true);
        }
        else
        {
            return new ContentLinkEntity(urlFactory.toRelativeUrlWithContextPath(entity.getLinkUrl()), entity.getLinkLabel(), "", true);
        }
    }
}
